import type { FC } from 'react';
import { useState } from 'react';
import styled from 'styled-components';
import {
  ExtendInteractionMedium,
  InteractionType,
  useInteractionTracking,
} from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { Button, Layout, Overflow, SVGAsset } from 'twitch-core-ui';
import { useAddToHomeScreenCTA } from '../AddToHomeScreen';
import { UpsellSheet } from '../UpsellSheet';

const INTERACTION_MEDIUM = 'follow_button_upsell';
const BRANCH_CAMPAIGN = 'follow_button_upsell';

export interface FollowButtonUpsellProps {
  displayName: string;
}

const ScSpan = styled.span`
  font-weight: bold;
`;

export const FollowButtonUpsell: FC<FollowButtonUpsellProps> = ({
  displayName,
}) => {
  const trackInteraction = useInteractionTracking();
  const { formatMessage } = useIntl();
  const [followUpsellOpen, setFollowUpsellOpen] = useState(false);
  const { enableShowCTA } = useAddToHomeScreenCTA();

  const upsellTitle = formatMessage(
    'Open the app and follow {displayName}!',
    {
      displayName: <ScSpan>{displayName}</ScSpan>,
    },
    'FollowButtonUpsell',
  );
  const upsellDescription = formatMessage(
    'Follow to get notified when {displayName} is live and so you don’t miss out on the action!',
    {
      displayName: <ScSpan>{displayName}</ScSpan>,
    },
    'FollowButtonUpsell',
  );

  const upsellButtonText = formatMessage('Open App', 'FollowButtonUpsell');

  function onFollowClick(): void {
    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: 'follow',
    });
    setFollowUpsellOpen(true);
    enableShowCTA();
  }

  return (
    <ExtendInteractionMedium value={INTERACTION_MEDIUM}>
      <>
        <Layout flexShrink={0} overflow={Overflow.Hidden}>
          <Button
            fullWidth
            icon={SVGAsset.FollowHollow}
            onClick={onFollowClick}
          >
            {formatMessage('Follow', 'FollowButton')}
          </Button>
        </Layout>
        {followUpsellOpen && (
          <UpsellSheet
            branchCampaign={BRANCH_CAMPAIGN}
            buttonText={upsellButtonText}
            callToActionIdentifier="follow_button"
            description={upsellDescription}
            onClose={() => {
              setFollowUpsellOpen(false);
            }}
            title={upsellTitle}
          />
        )}
      </>
    </ExtendInteractionMedium>
  );
};
