import type { FC } from 'react';
import { useCallback, useRef } from 'react';
import ReactList from 'react-list';
import { isBrowser, once, usePrevious } from 'tachyon-utils';

export type InfiniteListProps = {
  className?: string;
  itemRenderer: (index: number) => JSX.Element;
  length: number;
  loadMore: () => void;
  pageSize: number;
};

/**
 * Renders an infinite list. NOTE: The content rendered by itemRenderer must have the exact same dimensions -- variably sized collections may not be used.
 *
 * Read more: https://github.com/caseywebdev/react-list#type-one-of-simple-variable-or-uniform-defaults-to-simple
 */
export const InfiniteList: FC<InfiniteListProps> = ({
  className,
  itemRenderer,
  length,
  loadMore,
  pageSize,
}) => {
  const prevLength = usePrevious(length);
  const loadMoreOnce = useRef(once(loadMore));
  if (length !== prevLength) {
    loadMoreOnce.current = once(loadMore);
  }

  const autoLoadMoreItemRenderer = useCallback(
    (index: number): JSX.Element => {
      // loads more items if the user only has two pages of content left
      if (isBrowser() && length - index <= pageSize * 2) {
        loadMoreOnce.current();
      }
      return itemRenderer(index);
    },
    [itemRenderer, length, loadMoreOnce, pageSize],
  );

  const itemsRenderer = useCallback(
    (items: JSX.Element[], ref: string): JSX.Element => {
      return (
        // eslint-disable-next-line react/forbid-dom-props
        <div className={className} ref={ref}>
          {items}
        </div>
      );
    },
    [className],
  );

  return (
    <ReactList
      itemRenderer={autoLoadMoreItemRenderer}
      itemsRenderer={itemsRenderer}
      length={length}
      minSize={pageSize}
      type="uniform"
      // Do not use static size since the CSS will change the size of items
      // and the number of items in a row when the device is rotated.
      useStaticSize={false}
      useTranslate3d
    />
  );
};

InfiniteList.displayName = 'InfiniteList';
