import type { FC } from 'react';
import { useState } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { formatOffline } from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AlignItems,
  Avatar,
  Background,
  Button,
  ButtonSize,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  Position,
  TextAlign,
  TextType,
  ZIndex,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../routing';
import { useDeepLink } from '../../../utils';
import type { OfflineChannelUpsell_user } from './__generated__/OfflineChannelUpsell_user.graphql';

const BRANCH_CAMPAIGN = 'offline_channel_upsell';
const CALL_TO_ACTION_IDENTIFIER = `${BRANCH_CAMPAIGN}_cta`;

const ScBackgroundCover = styled(Layout)`
  opacity: 0.95;
`;

const ScAvatarContainer = styled.div`
  max-width: 80%;
`;

export type OfflineChannelUpsellBaseProps = { user: OfflineChannelUpsell_user };

export const OfflineChannelUpsellBase: FC<OfflineChannelUpsellBaseProps> = ({
  user,
}) => {
  const { formatMessage } = useIntl();
  const [dismissed, setDismissed] = useState(false);

  const url = useDeepLink(BRANCH_CAMPAIGN, CALL_TO_ACTION_IDENTIFIER);

  if (!url || dismissed) {
    return null;
  }

  return (
    <Layout
      attachBottom
      attachLeft
      attachRight
      attachTop
      display={Display.Flex}
      justifyContent={JustifyContent.Center}
      padding={1}
      position={Position.Fixed}
      zIndex={ZIndex.Above}
    >
      <ScBackgroundCover
        attachBottom
        attachLeft
        attachRight
        attachTop
        background={Background.Base}
        position={Position.Absolute}
      />
      <Layout
        alignItems={AlignItems.Center}
        background={Background.Base}
        display={Display.Flex}
        elevation={5}
        flexDirection={FlexDirection.Column}
        fullWidth
        justifyContent={JustifyContent.Center}
        margin="auto"
        padding={{ bottom: 2, top: 3, x: 1 }}
        textAlign={TextAlign.Center}
        zIndex={ZIndex.Above}
      >
        <Layout margin={{ y: 1 }}>
          <ScAvatarContainer>
            <Avatar
              alt=""
              size={96}
              src={user.largeProfileImageURL}
              userLogin={user.login}
            />
          </ScAvatarContainer>
        </Layout>
        <Layout margin={{ y: 0.5 }}>
          <CoreText color={Color.Base} type={TextType.H3}>
            {formatOffline(
              formatMessage,
              getFormattedUserDisplayName(user) || '',
            )}
          </CoreText>
        </Layout>
        <Layout margin={{ x: 1, y: 0.5 }}>
          <CoreText color={Color.Base} fontSize={FontSize.Size5}>
            {formatMessage(
              "Don't miss out on their next stream! Download the app to get notified the next time {user} goes live.",
              {
                user: (
                  <CoreText bold type={TextType.Span}>
                    {getFormattedUserDisplayName(user) || ''}
                  </CoreText>
                ),
              },
              'OfflineChannelUpsell',
            )}
          </CoreText>
        </Layout>
        <Layout fullWidth margin={{ y: 1 }}>
          <Button
            fullWidth
            linkTo="/deferToRenderLink"
            renderLink={renderTachyonLink({
              interactionContent: 'cta_button',
              route: RouteName.External,
              routeParams: { href: url },
            })}
            size={ButtonSize.Large}
          >
            {formatMessage('Get the Twitch App', 'OfflineChannelUpsell')}
          </Button>
        </Layout>
        <Layout margin={{ y: 0.5 }}>
          <Button
            onClick={() => {
              setDismissed(true);
            }}
            variant={ButtonType.Text}
          >
            {formatMessage('Not now', 'OfflineChannelUpsell')}
          </Button>
        </Layout>
      </Layout>
    </Layout>
  );
};

OfflineChannelUpsellBase.displayName = 'OfflineChannelUpsellBase';

export const OfflineChannelUpsell = createFragmentContainer(
  OfflineChannelUpsellBase,
  {
    user: graphql`
      fragment OfflineChannelUpsell_user on User {
        displayName
        login
        largeProfileImageURL: profileImageURL(width: 300)
      }
    `,
  },
);
