import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import {
  Avatar,
  StreamType,
  StreamTypeIndicator,
  formatLiveViewersCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  CoreImage,
  MediaCard,
  MediaCardImage,
  MediaCardLink,
  MediaCardMeta,
  MediaCardStat,
  MediaCardTitle,
} from 'twitch-core-ui';
import { StreamTagList } from '../tags';
import type { StreamCard_stream } from './__generated__/StreamCard_stream.graphql';

type StreamCardProps = {
  stream: StreamCard_stream;
  streamType?: StreamType;
};

// istanbul ignore next: trivial
export const StreamCardBase: FC<StreamCardProps> = ({
  stream,
  streamType = StreamType.Live,
}) => {
  const { formatMessage, formatNumberShort } = useIntl();

  const gameName = stream.game?.name ?? '';

  return (
    <MediaCard
      image={
        <MediaCardImage
          bottomLeft={
            stream.viewersCount !== null && (
              <MediaCardStat>
                {formatLiveViewersCount(
                  formatMessage,
                  stream.viewersCount,
                  formatNumberShort(stream.viewersCount, {
                    maximumSignificantDigits: 2,
                  }),
                )}
              </MediaCardStat>
            )
          }
          image={
            <CoreImage
              {...getChannelThumbnailSrcAndSrcSet(stream.previewImageURL)}
              alt=""
            />
          }
          topLeft={
            <StreamTypeIndicator
              isOverlay
              isPulsing={false}
              streamType={streamType}
            />
          }
        />
      }
      meta={
        <MediaCardMeta
          actions={
            <StreamTagList
              gameAlias={gameName}
              page={TagPage.Stream}
              stream={stream}
            />
          }
          image={
            stream.broadcaster ? (
              <Avatar
                alt=""
                sizeRem={3.6}
                src={stream.broadcaster.profileImageURL}
                userLogin={stream.broadcaster.login}
              />
            ) : null
          }
          links={
            <>
              {stream.broadcaster ? (
                <MediaCardLink>
                  {getFormattedUserDisplayName(stream.broadcaster)}
                </MediaCardLink>
              ) : null}
              <MediaCardLink>{gameName}</MediaCardLink>
            </>
          }
          title={
            <MediaCardTitle>
              {stream.broadcaster?.broadcastSettings?.title ?? ''}
            </MediaCardTitle>
          }
        />
      }
    />
  );
};

StreamCardBase.displayName = 'StreamCard';

export const StreamCard = createFragmentContainer(StreamCardBase, {
  stream: graphql`
    fragment StreamCard_stream on Stream {
      ...TagList_stream
      id
      previewImageURL
      viewersCount
      broadcaster {
        id
        broadcastSettings {
          title
        }
        login
        displayName
        profileImageURL(width: 50)
      }
      game {
        name
      }
    }
  `,
});
