import type { MouseEvent, MouseEventHandler, ReactNode } from 'react';
import { PureComponent } from 'react';
import {
  AlignItems,
  Animation,
  AnimationDelay,
  AnimationDuration,
  AnimationTiming,
  AnimationType,
  Color,
  Display,
  Layout,
  Position,
  Toast,
  ZIndex,
} from 'twitch-core-ui';

export interface TimedToastProps {
  children: ReactNode;
  onClickClose?: MouseEventHandler<HTMLElement>;
  secondsDelayUntilVisible?: number;
  secondsVisible: number;
}

interface TimedToastState {
  enabled: boolean;
  readyForFirstRender: boolean;
}

export class TimedToast extends PureComponent<
  TimedToastProps,
  TimedToastState
> {
  public static displayName = 'TimedToast';

  public state: TimedToastState = {
    enabled: false,
    readyForFirstRender: false,
  };

  private showTimerHandle: number | undefined = undefined;
  private hideTimerHandle: number | undefined = undefined;

  public componentDidMount(): void {
    this.showTimerHandle = window.setTimeout(() => {
      this.showToastAndStartHideTimer();
    }, (this.props.secondsDelayUntilVisible ?? 0) * 1000);
  }

  public componentWillUnmount(): void {
    this.clearTimers();
  }

  public render(): JSX.Element | false {
    return (
      this.state.readyForFirstRender && (
        <Layout
          alignItems={AlignItems.Center}
          fullWidth
          position={Position.Absolute}
          zIndex={ZIndex.Default}
        >
          <Animation
            delay={
              this.state.enabled ? AnimationDelay.Long : AnimationDelay.Short
            }
            duration={
              this.state.enabled
                ? AnimationDuration.Long
                : AnimationDuration.Medium
            }
            enabled
            timing={AnimationTiming.EaseIn}
            type={this.animationType}
          >
            {/* This works because the flex stylings override the width of the toast.*/}
            <Layout
              color={Color.Alt2}
              display={Display.Flex}
              flexGrow={1}
              fullWidth
            >
              <Toast onClose={this.clickClose}>{this.props.children}</Toast>
            </Layout>
          </Animation>
        </Layout>
      )
    );
  }

  private get animationType(): AnimationType {
    return this.state.enabled
      ? AnimationType.SlideInTop
      : AnimationType.SlideOutTop;
  }

  private showToastAndStartHideTimer(): void {
    this.setState(() => ({ enabled: true, readyForFirstRender: true }));

    this.hideTimerHandle = window.setTimeout(() => {
      this.setState(() => ({ enabled: false }));
    }, this.props.secondsVisible * 1000);
  }

  private clearTimers(): void {
    if (this.showTimerHandle) {
      clearTimeout(this.showTimerHandle);
    }

    if (this.hideTimerHandle) {
      clearTimeout(this.hideTimerHandle);
    }
  }

  private clickClose = (e: MouseEvent<HTMLButtonElement>): void => {
    if (this.props.onClickClose) {
      this.props.onClickClose(e);
    }

    this.setState(() => ({ enabled: false }));
    this.clearTimers();
  };
}
