import type { FC } from 'react';
import { useCallback, useMemo, useState } from 'react';
import { useCurrentUser } from 'tachyon-auth';
import { useDynamicSettings } from 'tachyon-dynamic-settings';
import { useStaticEnvironment } from 'tachyon-environment';
import {
  ExtendInteractionMedium,
  InteractionType,
  useInteractionTracking,
} from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import {
  TwitchEntityInformationLink,
  TwitchEntityInformationModal,
} from 'tachyon-more-ui';
import { useRequestInfo } from 'tachyon-relay';
import { useJSEnabled } from 'tachyon-utils';
import {
  Balloon,
  BalloonSize,
  Button,
  ButtonIcon,
  ButtonIconSize,
  ButtonType,
  Display,
  Layout,
  Pill,
  SVGAsset,
  useDialogState,
} from 'twitch-core-ui';
import type { TomorrowDynamicSettings } from '../../../../config';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { useTheme } from '../../../../theme';
import { useAddToHomeScreen } from '../../AddToHomeScreen';
import type { MenuItemProps } from '../TopNavMenuItem';
import { MenuItem, MenuItemType } from '../TopNavMenuItem';

export const KOREA_ISO_COUNTRY_CODE = 'KR';

export const INTERACTION_MEDIUM_MENU = 'menu';
export const INTERACTION_CONTENT_MENU = 'menu';
export const INTERACTION_CONTENT_PRIVACY_POLICY = 'privacy_policy';
export const INTERACTION_CONTENT_COOKIE_POLICY = 'cookie_policy';
export const INTERACTION_CONTENT_TOS = 'terms_of_service';
export const INTERACTION_CONTENT_ENTITY_INFORMATION = 'entity_information';
export const INTERACTION_CONTENT_SWITCH_TO_DESKTOP = 'switch_to_desktop';
export const INTERACTION_CONTENT_THEME_TO_LIGHT = 'theme_to_light';
export const INTERACTION_CONTENT_THEME_TO_DARK = 'theme_to_dark';
export const INTERACTION_CONTENT_INSTALL_WEB_APP = 'install_web_app';

export const TopNavMenu: FC = () => {
  const { formatMessage } = useIntl();
  const requestInfo = useRequestInfo();
  const { currentTheme, toggleTheme } = useTheme();
  const trackEvent = useInteractionTracking();
  const enabled = useJSEnabled();
  const {
    common: { appEnvironment },
  } = useStaticEnvironment();
  const { canInstall, install } = useAddToHomeScreen();
  const { loggedIn, logout } = useCurrentUser();

  const {
    close: closeMenu,
    dialogProps: menuDialogProps,
    setAnchorRef: setMenuAnchorRef,
    toggle: toggleMenu,
  } = useDialogState();

  const { cookiePolicyUrl, privacyPolicyUrl, termsOfServiceUrl } =
    useDynamicSettings<TomorrowDynamicSettings>();

  const [showKoreanEntityModal, setShowKoreanEntityModal] = useState(false);
  const showKoreanEntityInfo =
    requestInfo?.countryCode === KOREA_ISO_COUNTRY_CODE;

  const onMenuIconClick = useCallback(() => {
    toggleMenu();
    trackEvent({
      interaction: InteractionType.Click,
      interactionContent: INTERACTION_CONTENT_MENU,
    });
  }, [toggleMenu, trackEvent]);

  const toggleKoreanEntityModal = useCallback(() => {
    closeMenu();
    setShowKoreanEntityModal((prevState) => !prevState);
  }, [closeMenu]);

  const menuItems: MenuItemProps[] = useMemo(() => {
    const items: MenuItemProps[] = [
      {
        interactionContent: INTERACTION_CONTENT_PRIVACY_POLICY,
        text: formatMessage('Privacy Policy', 'TopNavMenu'),
        type: MenuItemType.Link,
        url: privacyPolicyUrl,
      },
      {
        interactionContent: INTERACTION_CONTENT_COOKIE_POLICY,
        text: formatMessage('Cookie Policy', 'TopNavMenu'),
        type: MenuItemType.Link,
        url: cookiePolicyUrl,
      },
      {
        interactionContent: INTERACTION_CONTENT_TOS,
        text: formatMessage('Terms of Service', 'TopNavMenu'),
        type: MenuItemType.Link,
        url: termsOfServiceUrl,
      },
    ];

    if (showKoreanEntityInfo) {
      items.push({
        interactionContent: INTERACTION_CONTENT_ENTITY_INFORMATION,
        onClick: toggleKoreanEntityModal,
        text: formatMessage('Entity Information', 'TopNavMenu'),
        type: MenuItemType.Link,
      });
    }

    items.push(
      {
        type: MenuItemType.Separator,
      },
      currentTheme === 'dark'
        ? {
            interactionContent: INTERACTION_CONTENT_THEME_TO_LIGHT,
            onClick: toggleTheme,
            text: formatMessage('Toggle Light Theme', 'TopNavMenu'),
            type: MenuItemType.Link,
          }
        : {
            interactionContent: INTERACTION_CONTENT_THEME_TO_DARK,
            onClick: toggleTheme,
            text: formatMessage('Toggle Dark Theme', 'TopNavMenu'),
            type: MenuItemType.Link,
          },
    );

    // Enables local development login by piggy backing off of Twilight's login. This is temporary while mobile web doesn't have a login UI.
    //
    // This is a convenience for testing embedded login pages
    // which in production have the web view's cookie jar seeded by the embedding native client.
    //
    // This isn't enabled in staging because our staging domain's don't match the scope set by Twilight's cookies *twitch.tv.
    //
    // Remove when TMW has a login UI.
    if (appEnvironment === 'development') {
      if (loggedIn) {
        items.push({
          interactionContent: '',
          onClick: logout,
          text: 'Logout (Dev Only)',
          type: MenuItemType.Link,
        });
      } else {
        items.push({
          interactionContent: '',
          onClick: () => {
            // eslint-disable-next-line no-alert
            window.alert(
              'Navigate back to mobile web via the back button after logging in.',
            );
          },
          text: 'Login (Dev Only)',
          type: MenuItemType.Link,
          url: 'https://www.twitch.tv/login',
        });
      }
    }

    if (canInstall) {
      items.push(
        {
          type: MenuItemType.Separator,
        },
        {
          interactionContent: INTERACTION_CONTENT_INSTALL_WEB_APP,
          onClick: () => {
            closeMenu();
            install();
          },
          text: (
            <>
              <Pill label={formatMessage('NEW', 'TopNavMenu')} />
              <Layout display={Display.Inline} padding={{ left: 0.5 }}>
                {formatMessage(
                  'Install the web app for faster fun!',
                  'TopNavMenu',
                )}
              </Layout>
            </>
          ),
          type: MenuItemType.Link,
        },
      );
    }

    return items;
  }, [
    appEnvironment,
    canInstall,
    closeMenu,
    cookiePolicyUrl,
    currentTheme,
    formatMessage,
    install,
    loggedIn,
    logout,
    privacyPolicyUrl,
    showKoreanEntityInfo,
    toggleKoreanEntityModal,
    toggleTheme,
    termsOfServiceUrl,
  ]);

  const linkProps = useMemo(
    () => ({
      businessInfo: {
        renderLink: renderTachyonLink({
          route: RouteName.External,
          routeParams: {
            href: TwitchEntityInformationLink.BusinessInfo,
          },
        }),
      },
      help: {
        renderLink: renderTachyonLink({
          route: RouteName.External,
          routeParams: { href: TwitchEntityInformationLink.Help },
        }),
      },
      termsOfSale: {
        renderLink: renderTachyonLink({
          route: RouteName.External,
          routeParams: {
            href: TwitchEntityInformationLink.TermsOfSale,
          },
        }),
      },
    }),
    [],
  );

  return (
    <>
      <TwitchEntityInformationModal
        formatMessage={formatMessage}
        linkProps={linkProps}
        onToggleModal={toggleKoreanEntityModal}
        show={showKoreanEntityModal}
      />
      <Layout margin={{ left: 0.5 }}>
        <ButtonIcon
          aria-label={formatMessage('Show top navigation menu', 'TopNav')}
          disabled={!enabled}
          icon={SVGAsset.NavMore}
          onClick={onMenuIconClick}
          ref={setMenuAnchorRef}
          size={ButtonIconSize.Large}
        />
        <Balloon size={BalloonSize.Small} {...menuDialogProps}>
          <ExtendInteractionMedium value={INTERACTION_MEDIUM_MENU}>
            <Layout padding={0.5}>
              {menuItems.map((item, idx) => (
                <MenuItem key={idx} {...item} />
              ))}
              <Button disabled variant={ButtonType.Text}>
                {formatMessage(
                  'Build ID: {buildId}',
                  { buildId: process.env.BUILD_ID },
                  'SettingsPage',
                )}
              </Button>
            </Layout>
          </ExtendInteractionMedium>
        </Balloon>
      </Layout>
    </>
  );
};

TopNavMenu.displayName = 'TopNavMenu';
