import { TwitchEntityInformationModal } from 'tachyon-more-ui';
import { useRequestInfo } from 'tachyon-relay';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { useTheme } from '../../../../theme';
import { useAddToHomeScreen } from '../../AddToHomeScreen';
import {
  INTERACTION_CONTENT_ENTITY_INFORMATION,
  INTERACTION_CONTENT_INSTALL_WEB_APP,
  INTERACTION_CONTENT_THEME_TO_DARK,
  INTERACTION_CONTENT_THEME_TO_LIGHT,
  KOREA_ISO_COUNTRY_CODE,
  TopNavMenu,
} from '.';

jest.mock('tachyon-relay', () => ({
  ...jest.requireActual('tachyon-relay'),
  useRequestInfo: jest.fn(),
}));
const mockUseRequestInfo = useRequestInfo as jest.Mock;

jest.mock('../../../../theme', () => ({
  ...jest.requireActual('../../../../theme'),
  useTheme: jest.fn(() => ({ currentTheme: '', toggleTheme: () => undefined })),
}));
const mockUseTheme = useTheme as jest.Mock;

jest.mock('../../AddToHomeScreen', () => ({
  ...jest.requireActual('../../AddToHomeScreen'),
  useAddToHomeScreen: jest.fn(
    jest.requireActual('../../AddToHomeScreen').useAddToHomeScreen,
  ),
}));
const mockUseAppInstall = useAddToHomeScreen as jest.Mock;

describe(TopNavMenu, () => {
  const setup = createShallowWrapperFactory(TopNavMenu);

  describe('Korean Entity Information', () => {
    it('does not appear when requestInfo is not yet available', () => {
      expect(
        setup().wrapper.find({
          interactionContent: INTERACTION_CONTENT_ENTITY_INFORMATION,
        }),
      ).not.toExist();
    });

    it('does not appear when country code does not match Korea', () => {
      mockUseRequestInfo.mockImplementationOnce(() => ({
        countryCode: 'US',
      }));

      expect(
        setup().wrapper.find({
          interactionContent: INTERACTION_CONTENT_ENTITY_INFORMATION,
        }),
      ).not.toExist();
    });

    it('appears when country code matches Korea', () => {
      mockUseRequestInfo.mockImplementationOnce(() => ({
        countryCode: KOREA_ISO_COUNTRY_CODE,
      }));

      expect(
        setup().wrapper.find({
          interactionContent: INTERACTION_CONTENT_ENTITY_INFORMATION,
        }),
      ).toExist();
    });

    it('opens the KFTC modal on button click', () => {
      mockUseRequestInfo.mockImplementationOnce(() => ({
        countryCode: KOREA_ISO_COUNTRY_CODE,
      }));

      const { wrapper } = setup();
      expect(wrapper.find(TwitchEntityInformationModal)).toHaveProp({
        show: false,
      });
      wrapper
        .find({ interactionContent: INTERACTION_CONTENT_ENTITY_INFORMATION })
        .simulate('click');
      expect(wrapper.find(TwitchEntityInformationModal)).toHaveProp({
        show: true,
      });
    });
  });

  describe('theme handling', () => {
    it('has correct functioning link when current theme is dark', () => {
      const toggleTheme = jest.fn();
      mockUseTheme.mockImplementationOnce(() => ({
        currentTheme: 'dark',
        toggleTheme,
      }));

      const { wrapper } = setup();
      const themeLink = wrapper.find({
        interactionContent: INTERACTION_CONTENT_THEME_TO_LIGHT,
      });
      expect(themeLink).toHaveProp({ text: 'Toggle Light Theme' });
      expect(toggleTheme).not.toHaveBeenCalled();

      themeLink.simulate('click');
      expect(toggleTheme).toHaveBeenCalled();
    });

    it('has correct functioning link when current theme is light', () => {
      const toggleTheme = jest.fn();
      mockUseTheme.mockImplementationOnce(() => ({
        currentTheme: 'light',
        toggleTheme,
      }));

      const { wrapper } = setup();
      const themeLink = wrapper.find({
        interactionContent: INTERACTION_CONTENT_THEME_TO_DARK,
      });
      expect(themeLink).toHaveProp({ text: 'Toggle Dark Theme' });
      expect(toggleTheme).not.toHaveBeenCalled();

      themeLink.simulate('click');
      expect(toggleTheme).toHaveBeenCalled();
    });
  });

  describe('App Install', () => {
    it('shows app install button when installable', () => {
      mockUseAppInstall.mockImplementationOnce(() => ({
        canInstall: true,
      }));

      expect(
        setup().wrapper.find({
          interactionContent: INTERACTION_CONTENT_INSTALL_WEB_APP,
        }),
      ).toExist();
    });

    it('does not show otherwise', () => {
      expect(
        setup().wrapper.find({
          interactionContent: INTERACTION_CONTENT_INSTALL_WEB_APP,
        }),
      ).not.toExist();
    });
  });
});
