import type { EventHandler, FC, MouseEvent, ReactNode } from 'react';
import type { InteractionTrackingProps } from 'tachyon-event-tracker';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import type { InteractableProps } from 'twitch-core-ui';
import { Color, CoreText, Interactable, Layout } from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';

export enum MenuItemType {
  Link = 'link',
  Separator = 'separator',
}

interface BaseMenuItemData {
  hidden?: boolean;
  type: MenuItemType;
}

interface MenuLinkData
  extends BaseMenuItemData,
    Omit<InteractionTrackingProps, 'trackInteraction'> {
  bold?: boolean;
  interactionContent: string;
  onClick?: EventHandler<MouseEvent<HTMLAnchorElement | HTMLButtonElement>>;
  text: ReactNode;
  type: MenuItemType.Link;
  url?: string;
}

export interface MenuSeparatorData extends BaseMenuItemData {
  type: MenuItemType.Separator;
}

export type MenuItemProps = MenuLinkData | MenuSeparatorData;

export const MenuItem: FC<MenuItemProps> = (props) => {
  const trackEvent = useInteractionTracking();

  if (props.hidden) {
    return null;
  }

  if (props.type === MenuItemType.Separator) {
    return <Layout borderBottom margin={{ x: 1, y: 0.5 }} />;
  }

  const url = props.url;
  const interactableProps: InteractableProps | null = url
    ? {
        linkTo: '/deferToRenderLink',
        renderLink: renderTachyonLink({
          interactionContent: props.interactionContent,
          interactionTargetPath: props.interactionTargetPath,
          route: RouteName.External,
          routeParams: { href: url },
        }),
      }
    : null;

  const onMenuItemClick: MenuLinkData['onClick'] = (event) => {
    if (props.onClick) {
      props.onClick(event);
    }

    if (!url) {
      trackEvent({
        interaction: InteractionType.Click,
        interactionContent: props.interactionContent,
      });
    }
  };

  return (
    <Interactable {...interactableProps} onClick={onMenuItemClick}>
      <Layout color={Color.Link} padding={{ x: 1, y: 0.5 }}>
        <CoreText bold={props.bold}>{props.text}</CoreText>
      </Layout>
    </Interactable>
  );
};

MenuItem.displayName = 'MenuItem';
