import type { FC } from 'react';
import styled from 'styled-components';
import { useStaticEnvironment } from 'tachyon-environment';
import { ExtendInteractionMedium } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import { safeAreaHorizontalPaddingToRemoveInsets } from 'tachyon-more-ui';
import {
  AlignItems,
  Background,
  ButtonIcon,
  ButtonIconSize,
  CoreLink,
  Display,
  InjectLayout,
  JustifyContent,
  Layout,
  Position,
  SVG,
  SVGAsset,
  SVGType,
  ZIndex,
} from 'twitch-core-ui';
import { TOP_NAV_HEIGHT_PX } from '../../../config';
import { RouteName, renderTachyonLink } from '../../../routing';
import { showOpenInApp } from '../../../utils';
import { OpenInApp } from '../OpenInApp';
import { SearchButton } from './SearchButton';
import { TopNavMenu } from './TopNavMenu';

export const INTERACTION_CONTENT_LOGO = 'logo';
export const INTERACTION_CONTENT_MENU = 'menu';

const ScTopNavContainer = styled(Layout)`
  height: ${TOP_NAV_HEIGHT_PX}px;
  line-height: 1rem;
  ${safeAreaHorizontalPaddingToRemoveInsets}
`;

export const TopNav: FC = () => {
  const {
    client: { agentInfo },
  } = useStaticEnvironment();
  const { formatMessage } = useIntl();
  const displayOpenInApp = showOpenInApp(agentInfo);

  return (
    <ExtendInteractionMedium value="top_nav">
      <ScTopNavContainer
        alignItems={AlignItems.Center}
        as="nav"
        attachLeft
        attachRight
        attachTop
        background={Background.Base}
        display={Display.Flex}
        elevation={2}
        fullWidth
        position={Position.Absolute}
        zIndex={ZIndex.Above}
      >
        <InjectLayout padding={{ x: 1 }}>
          <CoreLink
            aria-label={formatMessage('Go to the Twitch home page', 'TopNav')}
            linkTo="/deferToRenderLink"
            renderLink={renderTachyonLink({
              interactionContent: INTERACTION_CONTENT_LOGO,
              route: RouteName.Homepage,
            })}
          >
            <SVG
              asset={SVGAsset.LogoGlitch}
              height={32}
              type={SVGType.Brand}
              width={32}
            />
          </CoreLink>
        </InjectLayout>
        <Layout display={Display.Flex} margin={{ right: 1 }}>
          <Layout margin={{ left: 0.5 }}>
            <ButtonIcon
              aria-label={formatMessage('Go to all games page', 'TopNav')}
              icon={SVGAsset.Browse}
              linkTo="/deferToRenderLink"
              renderLink={renderTachyonLink({
                route: RouteName.GamesDirectory,
              })}
              size={ButtonIconSize.Large}
            />
          </Layout>
        </Layout>
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexGrow={1}
          justifyContent={JustifyContent.End}
          margin={{ right: 1 }}
        >
          <Layout
            display={displayOpenInApp ? Display.InlineFlex : Display.Hide}
            margin={{ right: 1 }}
          >
            {displayOpenInApp && <OpenInApp callToActionIdentifier="top_nav" />}
          </Layout>
          <Layout padding={{ left: 1 }}>
            <SearchButton />
          </Layout>
          <TopNavMenu />
        </Layout>
      </ScTopNavContainer>
    </ExtendInteractionMedium>
  );
};

TopNav.displayName = 'TopNav';
