import type { FC } from 'react';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import {
  Background,
  Button,
  ButtonIcon,
  ButtonIconSize,
  ButtonSize,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  Position,
  SVGAsset,
  TextType,
  ZIndex,
} from 'twitch-core-ui';
import type { StringOrElement } from 'twitch-intl';
import { RouteName, renderTachyonLink } from '../../../routing';
import { useDeepLink } from '../../../utils';

export interface UpsellSheetProps {
  /**
   * What campaign this to associate with on branch.io
   */
  branchCampaign: string;
  buttonText: StringOrElement;
  callToActionIdentifier?: string;
  description?: StringOrElement;
  onClose: () => void;
  /**
   * Overrides "currentRouteName" as defined by the application router that
   * would otherwise be set as the location to open in app.
   */
  targetAppLocation?: string;
  title: StringOrElement;
}

export const UpsellSheet: FC<UpsellSheetProps> = ({
  branchCampaign,
  buttonText,
  callToActionIdentifier,
  description,
  onClose,
  targetAppLocation,
  title,
}) => {
  const { formatMessage } = useIntl();
  const trackInteraction = useInteractionTracking();
  const url = useDeepLink(
    branchCampaign,
    callToActionIdentifier ?? 'upsell_sheet',
    targetAppLocation,
  );

  function onCloseClick(): void {
    trackInteraction({
      interaction: InteractionType.Click,
      interactionContent: 'upsell_close',
    });

    onClose();
  }

  if (!url) {
    return null;
  }

  return (
    <Layout
      attachBottom
      attachLeft
      attachRight
      attachTop
      background={Background.Overlay}
      onClick={onCloseClick}
      position={Position.Fixed}
      zIndex={ZIndex.Above}
    >
      <Layout
        attachBottom
        attachLeft
        attachRight
        background={Background.AccentAlt2}
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        onClick={(e) => {
          e.stopPropagation();
        }}
        padding={{ bottom: 3, top: 2, x: 4 }}
        position={Position.Absolute}
      >
        <Layout attachRight attachTop padding={1} position={Position.Absolute}>
          <ButtonIcon
            aria-label={formatMessage('Close overlay', 'UpsellSheet')}
            icon={SVGAsset.Close}
            onClick={onCloseClick}
            overlay
            size={ButtonIconSize.Small}
          />
        </Layout>
        <CoreText type={TextType.H4}>{title}</CoreText>
        {description && (
          <Layout margin={{ top: 0.5 }}>
            <CoreText>{description}</CoreText>
          </Layout>
        )}
        <Layout margin={{ top: 2 }}>
          <Button
            fullWidth
            linkTo={url ? '/deferToRenderLink' : undefined}
            renderLink={renderTachyonLink({
              interactionContent: 'upsell_cta',
              route: RouteName.External,
              routeParams: { href: url },
            })}
            size={ButtonSize.Large}
          >
            {buttonText}
          </Button>
        </Layout>
      </Layout>
    </Layout>
  );
};

UpsellSheet.displayName = 'UpsellSheet';
