import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import type { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import {
  formatTotalViewCount,
  getBoxArtSrcAndSrcSet,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { formatVideoLength, getFormattedUserDisplayName } from 'tachyon-utils';
import {
  AspectRatio,
  CardImage,
  CoreImage,
  CoreLink,
  CoreLinkType,
  MediaCard,
  MediaCardImage,
  MediaCardLink,
  MediaCardMeta,
  MediaCardStat,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../routing';
import { VideoTagList } from '../tags';
import type { VideoCard_video } from './__generated__/VideoCard_video.graphql';

type VideoCardProps = {
  tagPage: TagPage;
  video: VideoCard_video;
};

// istanbul ignore next: trivial
export const VideoCardBase: FC<VideoCardProps> = ({ tagPage, video }) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      renderLink={renderTachyonLink({
        route: RouteName.Vod,
        routeParams: { videoID: video.id },
      })}
      variant={CoreLinkType.Inherit}
    >
      <MediaCard
        image={
          <MediaCardImage
            bottomLeft={
              video.viewCount ? (
                <MediaCardStat>
                  {formatTotalViewCount(
                    formatMessage,
                    video.viewCount,
                    formatNumberShort(video.viewCount, {
                      maximumSignificantDigits: 2,
                    }),
                  )}
                </MediaCardStat>
              ) : null
            }
            bottomRight={
              video.publishedAt ? (
                <MediaCardStat>
                  {formatPastRelativeDate(new Date(video.publishedAt))}
                </MediaCardStat>
              ) : null
            }
            image={
              <CardImage
                {...getChannelThumbnailSrcAndSrcSet(video.previewThumbnailURL)}
                alt={video.title ?? ''}
              />
            }
            ratio={AspectRatio.Aspect16x9}
            topLeft={
              video.lengthSeconds ? (
                <MediaCardStat>
                  {formatVideoLength(video.lengthSeconds)}
                </MediaCardStat>
              ) : null
            }
          />
        }
        meta={
          <MediaCardMeta
            actions={
              <VideoTagList
                gameAlias={video.game?.name}
                page={tagPage}
                video={video}
              />
            }
            image={
              video.game?.boxArtURL ? (
                <CoreImage
                  alt={video.game.displayName}
                  {...getBoxArtSrcAndSrcSet(video.game.boxArtURL)}
                />
              ) : null
            }
            links={
              <>
                {video.owner ? (
                  <MediaCardLink>
                    {getFormattedUserDisplayName(video.owner)}
                  </MediaCardLink>
                ) : null}
                {video.game?.displayName ? (
                  <MediaCardLink>{video.game.displayName}</MediaCardLink>
                ) : null}
              </>
            }
            title={video.title ?? ''}
          />
        }
      />
    </CoreLink>
  );
};

VideoCardBase.displayName = 'VideoCard';

export const VideoCard = createFragmentContainer(VideoCardBase, {
  video: graphql`
    fragment VideoCard_video on Video {
      id
      previewThumbnailURL
      title
      lengthSeconds
      viewCount
      publishedAt
      game {
        displayName
        boxArtURL
        name
      }
      owner {
        login
        displayName
      }
      ...TagList_video
    }
  `,
});
