import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import type { SpadeTagClickEvent } from 'tachyon-discovery';
import { TagEvent, TagPage, TagSection } from 'tachyon-discovery';
import { useCustomTracking } from 'tachyon-event-tracker';
import type { TagProps } from 'twitch-core-ui';
import { Tag, TagAction } from 'twitch-core-ui';
import { TAG_QUERY_PARAM_KEY } from '../../../../config';
import type { TachyonLinkPublicProps } from '../../../../routing';
import { RouteName, renderTachyonLink } from '../../../../routing';
import type { TagLink_tag } from './__generated__/TagLink_tag.graphql';

export interface TagLinkPublicProps
  extends Pick<TagProps, 'action' | 'aria-label'>,
    Pick<TachyonLinkPublicProps<any>, 'interactionContent'> {
  gameAlias?: string | null;
  page: TagPage;
  position: number;
}

export type TagLinkProps = TagLinkPublicProps & {
  tag: TagLink_tag;
};

export const TagLinkBase: FC<TagLinkProps> = ({
  action,
  'aria-label': ariaLabel,
  gameAlias,
  interactionContent,
  page,
  position: tag_position,
  tag,
}) => {
  const trackEvent = useCustomTracking<SpadeTagClickEvent>();

  const { id: tag_id, localizedDescription, localizedName, tagName } = tag;

  const query =
    action === TagAction.Remove
      ? undefined
      : {
          [TAG_QUERY_PARAM_KEY]: tag_id,
        };
  const isBrowse = page === TagPage.Browse;
  const renderLink = isBrowse
    ? gameAlias
      ? renderTachyonLink({
          interactionContent,
          query,
          route: RouteName.GameDirectory,
          routeParams: { gameAlias },
        })
      : renderTachyonLink({
          interactionContent,
          query,
          route: RouteName.GamesDirectory,
        })
    : renderTachyonLink({
        interactionContent,
        query,
        route: RouteName.GameDirectoryAll,
      });

  function trackClick(): void {
    trackEvent({
      dismiss: action === TagAction.Remove,
      event: TagEvent.Filter,
      focus: null,
      item_page: TagPage.Browse,
      search_event: null,
      section: isBrowse ? TagSection.Browse : TagSection.LiveChannels,
      tag_id,
      tag_position,
    });
  }

  return (
    <Tag
      action={action}
      aria-label={ariaLabel ?? localizedDescription}
      label={localizedName || tagName}
      linkTo="/deferToRenderLink"
      onClick={trackClick}
      renderLink={renderLink}
      title={localizedDescription}
    />
  );
};

TagLinkBase.displayName = 'TagLinkBase';

export const TagLink = createFragmentContainer(TagLinkBase, {
  tag: graphql`
    fragment TagLink_tag on Tag {
      id
      tagName
      localizedDescription
      localizedName
    }
  `,
});
