import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import type { TagPage } from 'tachyon-discovery';
import {
  AlignContent,
  Display,
  FlexDirection,
  FlexWrap,
  JustifyContent,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import type { TachyonLinkPublicProps } from '../../../../routing';
import { TagLink } from '../TagLink';
import type { TagList_game } from './__generated__/TagList_game.graphql';
import type { TagList_stream } from './__generated__/TagList_stream.graphql';
import type { TagList_video } from './__generated__/TagList_video.graphql';

type TagListPublicProps = Pick<
  TachyonLinkPublicProps<any>,
  'interactionContent'
> & {
  gameAlias?: string | null;
  page: TagPage;
};

type GameTagListProps = TagListPublicProps & {
  game: TagList_game;
};

type StreamTagListProps = TagListPublicProps & {
  stream: TagList_stream;
};

type VideoTagListProps = TagListPublicProps & {
  video: TagList_video;
};

export type TagListProps =
  | GameTagListProps
  | StreamTagListProps
  | VideoTagListProps;

const TAG_LIMIT = 2;

const ScLayout = styled(Layout)`
  height: 3rem;
`;

export const TagListBase: FC<TagListProps> = (props) => {
  const tags =
    'game' in props
      ? props.game.gameTags
      : 'stream' in props
      ? props.stream.streamTags
      : props.video.videoTags;

  if (!tags) {
    return null;
  }

  const { gameAlias, interactionContent, page } = props;

  return (
    <ScLayout
      alignContent={AlignContent.Start}
      display={Display.InlineFlex}
      ellipsis
      flexDirection={FlexDirection.Row}
      flexWrap={FlexWrap.Wrap}
      fullWidth
      justifyContent={JustifyContent.Start}
      overflow={Overflow.Hidden}
      padding={{ top: 0.5 }}
    >
      {tags.slice(0, TAG_LIMIT).map((tag, key) => (
        <Layout fullHeight key={key} margin={{ right: 1 }}>
          <TagLink
            gameAlias={gameAlias}
            interactionContent={interactionContent}
            page={page}
            position={key}
            tag={tag}
          />
        </Layout>
      ))}
    </ScLayout>
  );
};

TagListBase.displayName = 'TagListBase';

export const GameTagList = createFragmentContainer<GameTagListProps>(
  TagListBase,
  {
    game: graphql`
      fragment TagList_game on Game {
        gameTags: tags(limit: 10, tagType: CONTENT) {
          ...TagLink_tag
        }
      }
    `,
  },
);

export const StreamTagList = createFragmentContainer<StreamTagListProps>(
  TagListBase,
  {
    stream: graphql`
      fragment TagList_stream on Stream {
        streamTags: tags {
          ...TagLink_tag
        }
      }
    `,
  },
);

export const VideoTagList = createFragmentContainer<VideoTagListProps>(
  TagListBase,
  {
    video: graphql`
      fragment TagList_video on Video {
        videoTags: contentTags {
          ...TagLink_tag
        }
      }
    `,
  },
);
