import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import type {
  ButtonIconProps,
  ButtonIconSize,
  ButtonIconType,
  ButtonProps,
  ButtonSize,
  ButtonType,
} from 'twitch-core-ui';
import {
  Button,
  ButtonIcon,
  CoreText,
  FontSize,
  SVGAsset,
} from 'twitch-core-ui';
import { useWebShare } from '../useWebShare';

interface WebShareButtonOwnProps {
  iconOnly?: boolean;
  size?: ButtonIconSize | ButtonSize;
  text?: string;
  title?: string;
  type?: ButtonIconType | ButtonType;
  url?: string;
}

type PartialButtonProps = Omit<
  ButtonProps,
  'dropdown' | 'icon' | 'purchase' | 'size' | 'state'
>;

type PartialButtonIconprops = Omit<
  ButtonIconProps,
  'aria-label' | 'dropdown' | 'icon' | 'size' | 'statusAlertIcon'
>;

export const WEB_SHARE_BUTTON_HEIGHT = '5.5rem';

export type WebShareButtonProps = PartialButtonIconprops &
  PartialButtonProps &
  WebShareButtonOwnProps;

export const WebShareButton: FC<WebShareButtonProps> = ({
  children,
  iconOnly,
  text,
  title,
  url,
  ...restProps
}) => {
  const { formatMessage } = useIntl();
  const [doShare, canShare] = useWebShare({ text, title, url });
  if (!canShare) {
    return null;
  }

  const message = formatMessage('Share', 'WebShareButton');

  if (iconOnly) {
    // we're rendering an button icon here...
    return (
      <ButtonIcon
        aria-label={message}
        icon={SVGAsset.Share}
        onClick={() => doShare()}
        {...(restProps as PartialButtonIconprops)}
      />
    );
  }

  return (
    <Button
      icon={SVGAsset.Share}
      onClick={() => doShare()}
      {...(restProps as PartialButtonProps)}
    >
      <CoreText bold fontSize={FontSize.Size5}>
        {children ?? message}
      </CoreText>
    </Button>
  );
};

WebShareButton.displayName = 'WebShareButton';
