import { useContext } from 'react';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import { routeNameFromPathname } from '../../../../routing';
import { webShareContext } from '../WebShareRoot';
import { canShare, wrapUrlWithTracking } from '../utils';

const SHARE_ACTION = 'share';
const SHARE_SUCCESS = 'share_success';
const SHARE_CANCEL = 'share_cancel';

export function useWebShare(
  options?: ShareData,
  onShare?: () => void,
  onShareCancel?: (error: PromiseRejectionEvent) => void,
): [() => Promise<void>, boolean] {
  const { canonicalURI, titleText } = useContext(webShareContext);
  const trackInteraction = useInteractionTracking();
  const { currentPathname } = useRouterUtils();
  const routeName = routeNameFromPathname(currentPathname);

  const doShare = async () => {
    if (canShare()) {
      trackInteraction({
        interaction: InteractionType.Click,
        interactionContent: SHARE_ACTION,
      });

      const url = wrapUrlWithTracking(options?.url ?? canonicalURI, routeName);
      const title = options?.title ?? titleText;
      const text = options?.text ?? '';

      try {
        await navigator.share({ text, title, url });

        trackInteraction({
          interaction: InteractionType.Click,
          interactionContent: SHARE_SUCCESS,
        });

        onShare?.();
      } catch (err) {
        trackInteraction({
          interaction: InteractionType.Click,
          interactionContent: SHARE_CANCEL,
        });

        onShareCancel?.(err as PromiseRejectionEvent);
      }
    }
  };

  return [doShare, canShare()];
}
