import { act, renderHook } from '@testing-library/react-hooks';
import { InteractionType } from 'tachyon-event-tracker';
import { useWebShare } from '.';

let mockShareResolve: (() => void) | null = null;
let mockShareReject: (() => void) | null = null;
const mockShareFunction: jest.Mock = jest.fn(
  () =>
    new Promise<void>((resolve, reject) => {
      mockShareResolve = () => {
        resolve();
      };
      mockShareReject = () => {
        reject();
      };
    }),
);

const mockTrackInteraction = jest.fn();
jest.mock('tachyon-event-tracker', () => ({
  ...jest.requireActual('tachyon-event-tracker'),
  useInteractionTracking: () => mockTrackInteraction,
}));

jest.mock('../../../../routing', () => ({
  routeNameFromPathname: () => 'homepage',
}));

const UI_INTERACTION_SHARE = {
  interaction: InteractionType.Click,
  interactionContent: 'share',
};
const UI_INTERACTION_SHARE_SUCCESS = {
  interaction: InteractionType.Click,
  interactionContent: 'share_success',
};
const UI_INTERACTION_SHARE_CANCEL = {
  interaction: InteractionType.Click,
  interactionContent: 'share_cancel',
};

const DEFAULT_SHARE = {
  text: '',
  title: 'Twitch',
  url: 'https://www.twitch.tv?tt_medium=mobile_web_share&tt_content=homepage',
};

describe(useWebShare, () => {
  beforeAll(() => {
    navigator.share = mockShareFunction;
  });
  afterAll(() => {
    (navigator as any).share = undefined;
  });

  it('has the correct return value', () => {
    const { result } = renderHook(() => useWebShare());
    expect(result.current).toEqual([expect.any(Function), true]);
  });

  it('handles the default share success flow', async () => {
    const { result } = renderHook(() => useWebShare());

    expect(mockShareFunction).not.toHaveBeenCalled();
    expect(mockTrackInteraction).not.toHaveBeenCalled();

    act(() => {
      result.current[0]();
    });

    expect(mockShareFunction).toHaveBeenCalledTimes(1);
    expect(mockShareFunction).toHaveBeenLastCalledWith(DEFAULT_SHARE);
    expect(mockTrackInteraction).toHaveBeenCalledTimes(1);
    expect(mockTrackInteraction).toHaveBeenLastCalledWith(UI_INTERACTION_SHARE);

    mockShareResolve!();
    await Promise.resolve();

    expect(mockShareFunction).toHaveBeenCalledTimes(1);
    expect(mockTrackInteraction).toHaveBeenCalledTimes(2);
    expect(mockTrackInteraction).toHaveBeenLastCalledWith(
      UI_INTERACTION_SHARE_SUCCESS,
    );
  });

  it('handles the default share cancel flow', async () => {
    const { result } = renderHook(() => useWebShare());

    expect(mockShareFunction).not.toHaveBeenCalled();
    expect(mockTrackInteraction).not.toHaveBeenCalled();

    act(() => {
      result.current[0]();
    });

    expect(mockShareFunction).toHaveBeenCalledTimes(1);
    expect(mockShareFunction).toHaveBeenLastCalledWith(DEFAULT_SHARE);
    expect(mockTrackInteraction).toHaveBeenCalledTimes(1);
    expect(mockTrackInteraction).toHaveBeenLastCalledWith(UI_INTERACTION_SHARE);

    mockShareReject!();
    await Promise.resolve();

    expect(mockShareFunction).toHaveBeenCalledTimes(1);
    expect(mockTrackInteraction).toHaveBeenCalledTimes(2);
    expect(mockTrackInteraction).toHaveBeenLastCalledWith(
      UI_INTERACTION_SHARE_CANCEL,
    );
  });
});
