import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { formatVideoLength } from 'tachyon-utils';
import { CoreLink } from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../routing';
import { VideoRowItemCommon } from '../VideoRowItemCommon';
import type { ClipRowItem_clip } from './__generated__/ClipRowItem_clip.graphql';

export interface ClipRowItemProps {
  clip: ClipRowItem_clip;
  interactionContent: string;
}

export const ClipRowItemBase: FC<ClipRowItemProps> = ({
  clip,
  interactionContent,
}) => {
  const routeData = clip.broadcaster?.login
    ? {
        route: RouteName.Clip,
        routeParams: { login: clip.broadcaster.login, slug: clip.slug },
      }
    : { route: RouteName.ClipLegacy, routeParams: { slug: clip.slug } };

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      renderLink={renderTachyonLink({
        interactionContent,
        interactionTargetPath: `/clips/${clip.slug}`,
        ...routeData,
      })}
    >
      <VideoRowItemCommon
        createdAt={new Date(clip.createdAt)}
        formattedLength={
          clip.durationSeconds ? formatVideoLength(clip.durationSeconds) : null
        }
        gameName={clip.game?.name ?? null}
        imageURL={clip.thumbnailURL}
        title={clip.title}
        viewCount={clip.viewCount}
      />
    </CoreLink>
  );
};

ClipRowItemBase.displayName = 'ClipRowItemBase';

export const ClipRowItem = createFragmentContainer(ClipRowItemBase, {
  clip: graphql`
    fragment ClipRowItem_clip on Clip {
      id
      title
      thumbnailURL
      viewCount
      durationSeconds
      createdAt
      slug
      broadcaster {
        login
      }
      game {
        id
        name
      }
    }
  `,
});
