import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import { getBoxArtSrcAndSrcSet } from 'tachyon-more-ui';
import type { CoreImageSize, CoreLinkProps } from 'twitch-core-ui';
import {
  AspectRatio,
  Background,
  Card,
  CardBody,
  CardImage,
  Color,
  CoreLink,
  CoreText,
  FontSize,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import type { TachyonLinkPublicProps } from '../../../../routing';
import { RouteName, renderTachyonLink } from '../../../../routing';

export type GameCardPosterProps = Pick<
  TachyonLinkPublicProps<RouteName.GameDirectory>,
  'interactionContent'
> &
  Pick<CoreLinkProps, 'onClick'> & {
    boxArtURL: string | null;
    gameName: string;
    srcSetSizes: CoreImageSize[];
    tagContainer: JSX.Element | null;
    viewersCount: string;
  };

// istanbul ignore next: trivial
export const GameCardPoster: FC<GameCardPosterProps> = ({
  boxArtURL,
  gameName,
  interactionContent,
  onClick,
  srcSetSizes,
  tagContainer,
  viewersCount,
}) => {
  const { formatMessage } = useIntl();
  const linkToGame = renderTachyonLink({
    interactionContent,
    route: RouteName.GameDirectory,
    routeParams: { gameAlias: gameName },
  });

  return (
    <CoreLink
      linkTo="/deferToRenderLink"
      onClick={onClick}
      renderLink={linkToGame}
    >
      <Card background={Background.Base} elevation={1}>
        <CardImage
          {...getBoxArtSrcAndSrcSet(boxArtURL)}
          alt=""
          aspect={AspectRatio.BoxArt}
          key={gameName}
          sizes={srcSetSizes}
        />
        <CardBody>
          <Layout overflow={Overflow.Hidden} padding={{ x: 1, y: 0.5 }}>
            <CoreText
              bold
              color={Color.Base}
              ellipsis
              fontSize={FontSize.Size5}
            >
              {gameName}
            </CoreText>
            <CoreText color={Color.Alt} ellipsis fontSize={FontSize.Size6}>
              {formatMessage(
                '{viewersCount} viewers',
                { viewersCount },
                'GameViewersCountPoster',
              )}
            </CoreText>
          </Layout>
          <Layout padding={{ bottom: 0.5, x: 1 }}>{tagContainer}</Layout>
        </CardBody>
      </Card>
    </CoreLink>
  );
};

GameCardPoster.displayName = 'GameCardPoster';
