import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { strongText } from 'tachyon-more-ui';
import type { CoreImageSize, CoreLinkProps } from 'twitch-core-ui';
import type { TachyonLinkPublicProps } from '../../../routing';
import { GameTagList } from '../../common';
import { GameCardPoster } from './GameCardPoster';
import { GameCardRow } from './GameCardRow';
import type { GameCardCommon_game } from './__generated__/GameCardCommon_game.graphql';

export type { GameCardCommon_game } from './__generated__/GameCardCommon_game.graphql';

export type GameCardCommonProps = Pick<CoreLinkProps, 'onClick'> &
  Pick<TachyonLinkPublicProps<any>, 'interactionContent'> & {
    game: GameCardCommon_game;
    hideTags?: boolean;
    showMoreDetails?: boolean;
    srcSetSizes: CoreImageSize[];
  };

export const GameCardCommonBase: FC<GameCardCommonProps> = (
  props: GameCardCommonProps,
) => {
  const { formatMessage, formatNumberShort } = useIntl();

  const {
    game,
    hideTags,
    interactionContent,
    onClick,
    showMoreDetails,
    srcSetSizes,
  } = props;
  const { boxArtURL, name: gameAlias } = game;

  const tagContainer = !hideTags ? (
    <GameTagList game={game} page={TagPage.Browse} />
  ) : null;

  const viewersCount = formatNumberShort(game.viewersCount ?? 0, {
    maximumSignificantDigits: 3,
  });

  if (showMoreDetails) {
    const followersCount = formatNumberShort(game.followersCount ?? 0, {
      maximumSignificantDigits: 3,
    });

    return (
      <GameCardRow
        boxArtURL={boxArtURL}
        details={formatMessage(
          '<x:strong>{viewersCount}</x:strong> Viewers · <x:strong>{followersCount}</x:strong> Followers',
          {
            followersCount,
            viewersCount,
            'x:strong': strongText,
          },
          'GameViewersCount',
        )}
        gameName={gameAlias}
        interactionContent={interactionContent}
        srcSetSizes={srcSetSizes}
        tagContainer={tagContainer}
      />
    );
  }

  return (
    <GameCardPoster
      boxArtURL={boxArtURL}
      gameName={gameAlias}
      interactionContent={interactionContent}
      onClick={onClick}
      srcSetSizes={srcSetSizes}
      tagContainer={tagContainer}
      viewersCount={viewersCount}
    />
  );
};

GameCardCommonBase.displayName = 'GameCardCommonBase';

export const GameCardCommon = createFragmentContainer(GameCardCommonBase, {
  game: graphql`
    fragment GameCardCommon_game on Game {
      name
      displayName
      viewersCount
      followersCount
      boxArtURL
      ...TagList_game
    }
  `,
});
