import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { TagPage } from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import {
  StreamTypeIndicator,
  getChannelThumbnailSrcAndSrcSet,
  getStreamType,
} from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import type { CoreImageSize, CoreLinkProps } from 'twitch-core-ui';
import {
  AspectRatio,
  Avatar,
  Background,
  Card,
  CardBody,
  CardImage,
  Color,
  CoreLink,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Layout,
  Overflow,
  Position,
} from 'twitch-core-ui';
import { StaticTokenMap } from 'twitch-core-ui-tokens';
import type { TachyonLinkPublicProps } from '../../../routing';
import { RouteName, renderTachyonLink } from '../../../routing';
import { StreamTagList } from '../../common';
import type { StreamCardCommon_stream } from './__generated__/StreamCardCommon_stream.graphql';

export type { StreamCardCommon_stream } from './__generated__/StreamCardCommon_stream.graphql';

export const STREAM_CARD_METADATA_HEIGHT_PX = 67;

const ScPreviewContainer = styled.div`
  position: relative;

  &::after {
    background: linear-gradient(
      to top,
      ${StaticTokenMap['color-twitch-purple-3']},
      transparent
    );
    bottom: 0;
    content: '';
    height: 3.3rem;
    left: 0;
    position: absolute;
    right: 0;
  }
`;

const ScChannelStatusContainer = styled.div`
  color: ${StaticTokenMap['color-hinted-grey-15']};
  left: 1rem;
  position: absolute;
  top: 1rem;
`;

const ScViewerCountContainer = styled(Layout)`
  bottom: 1rem;
  color: ${StaticTokenMap['color-hinted-grey-15']};
  left: 1rem;
  position: absolute;
`;

export interface StreamCardCommonPublicProps
  extends Pick<CoreLinkProps, 'onClick'>,
    Pick<TachyonLinkPublicProps<RouteName.Channel>, 'interactionContent'> {
  hideTags?: boolean;
  srcSetSizes: CoreImageSize[];
}

export interface StreamCardCommonProps extends StreamCardCommonPublicProps {
  stream: StreamCardCommon_stream;
}

type Stream = NonNullable<StreamCardCommon_stream>;
interface RenderableStream extends Stream {
  broadcaster: Stream['broadcaster'] & { login: string };
}

function isFoundStream(
  stream: StreamCardCommon_stream,
): stream is RenderableStream {
  if (!stream.broadcaster?.login) {
    return false;
  }

  return true;
}

export const StreamCardCommonBase: FC<StreamCardCommonProps> = ({
  hideTags,
  interactionContent,
  onClick,
  srcSetSizes,
  stream,
}) => {
  const { formatMessage, formatNumberShort } = useIntl();

  if (!isFoundStream(stream)) {
    return null;
  }

  const { broadcaster, game, previewImageURL } = stream;
  const gameName = game?.name ?? '';
  const viewersCount = formatNumberShort(stream.viewersCount ?? 0, {
    maximumSignificantDigits: 3,
  });

  const streamType = getStreamType(stream);

  const link = renderTachyonLink({
    interactionContent,
    route: RouteName.Channel,
    routeParams: { login: broadcaster.login },
  });

  return (
    <CoreLink linkTo="/deferToRenderLink" onClick={onClick} renderLink={link}>
      <Card background={Background.Base} elevation={1}>
        <Layout position={Position.Relative}>
          <ScPreviewContainer>
            <CardImage
              {...getChannelThumbnailSrcAndSrcSet(previewImageURL)}
              alt=""
              aspect={AspectRatio.Aspect16x9}
              key={broadcaster.login}
              sizes={srcSetSizes}
            />
          </ScPreviewContainer>
          <ScChannelStatusContainer>
            <StreamTypeIndicator
              isOverlay
              isPulsing={false}
              streamType={streamType}
            />
          </ScChannelStatusContainer>
          <ScViewerCountContainer ellipsis>
            {formatMessage(
              '{viewersCount} viewers',
              { viewersCount },
              'StreamCardCommon',
            )}
          </ScViewerCountContainer>
        </Layout>
        <CardBody>
          <Layout
            display={Display.Flex}
            flexDirection={FlexDirection.Row}
            overflow={Overflow.Hidden}
            padding={{ x: 1 }}
          >
            <Layout flexShrink={0} margin={{ top: 1 }}>
              <Avatar
                alt=""
                size={36}
                src={broadcaster.profileImageURL}
                userLogin={broadcaster.login}
              />
            </Layout>
            <Layout
              display={Display.Flex}
              ellipsis
              flexDirection={FlexDirection.Column}
            >
              <Layout flexShrink={2} margin={{ left: 1, y: 0.5 }}>
                <CoreText
                  bold
                  color={Color.Base}
                  ellipsis
                  fontSize={FontSize.Size5}
                >
                  {broadcaster.broadcastSettings?.title ?? ''}
                </CoreText>
                <CoreText color={Color.Alt} ellipsis fontSize={FontSize.Size6}>
                  {getFormattedUserDisplayName(broadcaster)}
                </CoreText>
                <CoreText color={Color.Alt2} ellipsis fontSize={FontSize.Size6}>
                  {gameName}
                </CoreText>
              </Layout>
              {!hideTags && (
                <Layout padding={{ bottom: 0.5, x: 1 }}>
                  <StreamTagList
                    gameAlias={gameName}
                    page={TagPage.Stream}
                    stream={stream}
                  />
                </Layout>
              )}
            </Layout>
          </Layout>
        </CardBody>
      </Card>
    </CoreLink>
  );
};

StreamCardCommonBase.displayName = 'StreamCardCommonBase';

export const StreamCardCommon = createFragmentContainer(StreamCardCommonBase, {
  stream: graphql`
    fragment StreamCardCommon_stream on Stream {
      id
      viewersCount
      previewImageURL
      type
      game {
        id
        name
      }
      broadcaster {
        id
        broadcastSettings {
          title
        }
        login
        displayName
        profileImageURL(width: 50)
        hosting {
          id
        }
      }
      ...TagList_stream
    }
  `,
});
