import type { FC } from 'react';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { formatTotalViewCount, getOverlaySrcAndSrcSet } from 'tachyon-more-ui';
import {
  AlignItems,
  AspectRatio,
  Background,
  CardImage,
  Color,
  CoreText,
  Display,
  FlexWrap,
  JustifyContent,
  Layout,
  Position,
  SVGAsset,
  Stat,
  TextType,
} from 'twitch-core-ui';
import { StaticTokenMap } from 'twitch-core-ui-tokens';

interface VideoRowItemCommonProps {
  createdAt: Date | null;
  formattedLength: string | null;
  gameName: string | null;
  imageURL: string;
  title: string | null;
  viewCount: number | null;
}

const ScImageContainer = styled(Layout)`
  height: 60px;
  width: 105px;
`;

const ScTimeContainer = styled(Layout)`
  bottom: -4px;
  right: 4px;
  text-shadow: 2px 2px 5px ${StaticTokenMap['color-hinted-grey-3']};
`;

export const VideoRowItemCommon: FC<VideoRowItemCommonProps> = ({
  createdAt,
  formattedLength,
  gameName,
  imageURL,
  title,
  viewCount,
}) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();

  return (
    <Layout
      alignItems={AlignItems.Center}
      background={Background.Base}
      display={Display.Flex}
      flexWrap={FlexWrap.NoWrap}
      fullWidth
      justifyContent={JustifyContent.Start}
      padding={1}
    >
      <ScImageContainer flexShrink={0} position={Position.Relative}>
        <CardImage
          {...getOverlaySrcAndSrcSet(imageURL)}
          alt=""
          aspect={AspectRatio.Aspect16x9}
          key={imageURL}
          sizes={[{ size: '105px' }]}
        />
        {formattedLength && (
          <ScTimeContainer color={Color.Overlay} position={Position.Absolute}>
            <Stat
              icon={SVGAsset.GlyphLength}
              label={formatMessage('Video length', 'VideoRowItem')}
              value={formattedLength}
            />
          </ScTimeContainer>
        )}
      </ScImageContainer>
      <Layout ellipsis margin={{ left: 1 }}>
        <CoreText color={Color.Base} ellipsis type={TextType.H4}>
          {title}
        </CoreText>
        <CoreText color={Color.Alt2} ellipsis>
          {gameName}
          &nbsp;&#124;&nbsp;
          {formatTotalViewCount(
            formatMessage,
            viewCount ?? 0,
            formatNumberShort(viewCount ?? 0, { maximumSignificantDigits: 3 }),
          )}
          &nbsp;&#124;&nbsp;
          {createdAt ? formatPastRelativeDate(createdAt) : null}
        </CoreText>
      </Layout>
    </Layout>
  );
};

VideoRowItemCommon.displayName = 'VideoRowItemCommon';
