import Head from 'next/head';
import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useStaticEnvironment } from 'tachyon-environment';
import { JSONLD, Metatags } from 'tachyon-metadata';
import { useRouterUtils } from 'tachyon-next-routing-utils';
import type { TachyonPageContext } from 'tachyon-next-types';
import {
  BASE_CANONICAL_URL,
  BASE_MOBILE_URL,
  DEFAULT_TITLE,
} from '../../../config';
import { RouteName, routeNameFromPathname } from '../../../routing';
import { WebShareRoot } from '../../common';
import type { PageHead_query } from './__generated__/PageHead_query.graphql';
import { useAlternateLanguageTags } from './useAlternateLanguageTags';

/**
 *  Routes in this list will render a robot 'noindex' metatag, preventing search indexing
 */
export const NOINDEX_LIST = [
  RouteName.AppShell,
  RouteName.ChannelStandalone,
  RouteName.HealthCheck,
  RouteName.NotFound,
  RouteName.Upsell,
  RouteName.Upsell2,
];

export interface PageHeadProps {
  isOffline: boolean;
  loading: boolean;
  query: PageHead_query | undefined;
}

// We use an enum for our keys so that the generated head only has a single
// version of each tag.
export enum Keys {
  Canonical = 'crawler-info-canonical',
  Description = 'crawler-info-description',
  RobotsIndex = 'crawler-info-robots-index',
  Title = 'crawler-info-title',
}

export type PageHeadQueryVariables = {
  url: string;
};

export function pageHeadQueryVariables(
  context: TachyonPageContext,
): PageHeadQueryVariables {
  return {
    // isomorphic reconstruction of URL to enable server and client queries
    url: `${BASE_MOBILE_URL}${context.asPath ?? ''}`,
  };
}

/**
 * An element for populating head with the correct meta tags for search engine
 * crawler indexing.
 * Head can only track primitive components, so PageHead inside <Head>
 * doesn't work, it must contain a <Head> instead
 */
export const PageHeadBase: FC<PageHeadProps> = ({
  children,
  isOffline,
  loading,
  query,
}) => {
  const alternateLanguageTags = useAlternateLanguageTags();
  const { currentPathname } = useRouterUtils();
  const noIndex = NOINDEX_LIST.includes(routeNameFromPathname(currentPathname));
  const {
    common: { appEnvironment },
  } = useStaticEnvironment();
  const canonicalURI = query?.urlMetadata?.share.url ?? BASE_CANONICAL_URL;
  const title = query?.urlMetadata?.title ?? DEFAULT_TITLE;
  // https://developers.facebook.com/docs/applinks/metadata-reference
  // native apps parse the URL themselves now, so replicating that here
  // transform "https://www.twitch.tv/moonmoon" -> "twitch://www.twitch.tv/moonmoon"
  // "m.twitch.tv/moonmoon" only works on android **but not on ios**

  if (
    appEnvironment !== 'production' &&
    !loading &&
    !isOffline &&
    !noIndex &&
    !query
  ) {
    throw new Error(
      '[PageHead] The current page is missing a Twitch MetaDog query. All pages should have this query to populate meta tags for SEO. If you are certain this page should not be included in search engine indices, add it to the "NOINDEX_LIST".',
    );
  }

  return (
    <>
      <Head>
        {alternateLanguageTags.map((tag) => (
          <link
            href={tag.href}
            hrefLang={tag.code}
            key={tag.code}
            rel="alternate"
          />
        ))}
        <title>{title}</title>
        {noIndex && (
          <meta content="noindex" key={Keys.RobotsIndex} name="robots" />
        )}
        {!noIndex &&
          query?.urlMetadata?.metatags &&
          /* This calling syntax is necessary because of this Next constraint: https://nextjs.org/docs/api-reference/next/head */
          // eslint-disable-next-line new-cap
          Metatags({ metatags: query.urlMetadata.metatags })}
        {!noIndex &&
          query?.urlMetadata?.jsonld &&
          /* This calling syntax is necessary because of this Next constraint: https://nextjs.org/docs/api-reference/next/head */
          // eslint-disable-next-line new-cap
          JSONLD({ data: query.urlMetadata.jsonld })}
      </Head>
      <WebShareRoot canonicalURI={canonicalURI} titleText={title}>
        {children}
      </WebShareRoot>
    </>
  );
};

PageHeadBase.displayName = 'PageHeadBase';

export const PageHead = createFragmentContainer(PageHeadBase, {
  query: graphql`
    fragment PageHead_query on Query {
      urlMetadata(url: $url) {
        title
        metatags {
          name
          attributes {
            key
            value
          }
        }
        jsonld
        share {
          title
          text
          url
        }
      }
    }
  `,
});
