import { useRouterUtils } from 'tachyon-next-routing-utils';
import type { TachyonPageContext } from 'tachyon-next-types';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { BASE_MOBILE_URL } from '../../../config';
import { RouteLinkPathnames } from '../../../routing';
import { NOINDEX_LIST, PageHeadBase, pageHeadQueryVariables } from '.';

jest.mock('tachyon-next-routing-utils', () => ({
  useRouterUtils: jest.fn(),
}));
const mockRouterContext = { currentAsPath: '/', currentPathname: '/' };
const mockUseRouterUtils = useRouterUtils as jest.Mock;
mockUseRouterUtils.mockImplementation(() => mockRouterContext);

describe(PageHeadBase, () => {
  const setup = createShallowWrapperFactory(PageHeadBase, () => ({
    isOffline: false,
    loading: false,
    query: {
      ' $refType': 'PageHead_query',
      urlMetadata: null,
    },
  }));

  it('renders the hrefLang tags', () => {
    const { wrapper } = setup();
    expect(wrapper.find('link[hrefLang="fr"]')).toHaveProp({
      href: 'https://m.twitch.tv/?lang=fr',
    });
  });

  it('renders noindex when route is in the NOINDEX_LIST', () => {
    mockUseRouterUtils.mockImplementation(() => ({
      currentAsPath: '/',
      currentPathname: RouteLinkPathnames[NOINDEX_LIST[0]],
    }));
    const { wrapper } = setup({ query: undefined });

    expect(wrapper.find('meta[name="robots"]')).toHaveProp({
      content: 'noindex',
      name: 'robots',
    });

    mockUseRouterUtils.mockImplementation(() => mockRouterContext);
  });

  it('throws when page does not provide metadog query and not in the NOINDEX_LIST', () => {
    expect(() => setup({ query: undefined })).toThrow();
  });
});

describe(pageHeadQueryVariables, () => {
  it('generates url from asPath', () => {
    const asPath = '/yolo';
    const vars = pageHeadQueryVariables({
      asPath,
    } as TachyonPageContext);
    expect(vars.url).toEqual(BASE_MOBILE_URL + asPath);
  });

  it('use BASE_MOBILE_URL when asPath is nullish', () => {
    const vars = pageHeadQueryVariables({} as TachyonPageContext);
    expect(vars.url).toEqual(BASE_MOBILE_URL);
  });
});
