import { renderHook } from '@testing-library/react-hooks';
import { BranchEvent, useBranchTimedTracker } from '.';

jest.mock('tachyon-environment', () => {
  const original = jest.requireActual('tachyon-environment');
  return {
    ...original,
    useStaticEnvironment: original.mockUseStaticEnvironment(),
  };
});

const mockTrackFn = jest.fn();
jest.mock('react-branch', () => ({
  ...jest.requireActual('react-branch'),
  useTracking: () => ({ track: mockTrackFn }),
}));

describe(useBranchTimedTracker, () => {
  beforeEach(() => {
    mockTrackFn.mockReset();
  });

  it('tracks at the appropiate time', () => {
    expect(mockTrackFn).not.toHaveBeenCalled();
    renderHook(() => {
      useBranchTimedTracker(BranchEvent.FiveMinutePlay, 5 * 60);
    });
    expect(mockTrackFn).not.toHaveBeenCalled();

    jest.advanceTimersByTime(4 * 60 * 1000);

    expect(mockTrackFn).not.toHaveBeenCalled();

    jest.advanceTimersByTime(60 * 1000);
    expect(mockTrackFn).toHaveBeenCalledTimes(1);
    expect(mockTrackFn).toHaveBeenCalledWith(BranchEvent.FiveMinutePlay, {
      device_id: 'ios-safari-device',
      session_id: 'ios-safari-session',
    });
  });

  it('does not track if the hook gets unmounted before time is up', () => {
    expect(mockTrackFn).not.toHaveBeenCalled();
    const { unmount } = renderHook(() => {
      useBranchTimedTracker(BranchEvent.FiveMinutePlay, 5 * 60);
    });
    expect(mockTrackFn).not.toHaveBeenCalled();
    jest.advanceTimersByTime(4 * 60 * 1000);
    expect(mockTrackFn).not.toHaveBeenCalled();
    unmount();
    jest.advanceTimersByTime(60 * 1000);
    expect(mockTrackFn).not.toHaveBeenCalled();
  });
});
