import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import {
  getFormattedUserDisplayName,
  reduceToNonNullNodes,
  uniq,
} from 'tachyon-utils';
import { Color, CoreText } from 'twitch-core-ui';
import type { TwitchIntl } from 'twitch-intl';
import type { ChannelDescription_user } from './__generated__/ChannelDescription_user.graphql';

type ChannelDescriptionProps = {
  expanded?: boolean;
  user: ChannelDescription_user;
};

const getFormattedBio = (
  { formatMessage }: TwitchIntl,
  user: ChannelDescription_user,
) => {
  const bio = user.description;
  if (bio?.length) {
    return bio;
  }

  const lastVideos = reduceToNonNullNodes(user.videos?.edges).map((node) => {
    return node.game?.displayName;
  });

  const allGames = [
    user.lastBroadcast?.game?.displayName,
    ...lastVideos,
  ].filter(Boolean) as string[];

  const games = uniq(allGames);

  const [firstGame, secondGame, thirdGame] = games;
  const displayName = getFormattedUserDisplayName(user);

  if (games.length === 0) {
    return formatMessage(
      "We don't know much about them, but we're sure {user} is great.",
      { user: displayName },
      'ChannelDescription',
    );
  } else if (games.length === 1) {
    return formatMessage(
      '{user} streams {firstGame}.',
      { firstGame, user: displayName },
      'ChannelDescription',
    );
  } else if (games.length === 2) {
    return formatMessage(
      '{user} streams {firstGame} and {secondGame}.',
      { firstGame, secondGame, user: displayName },
      'ChannelDescription',
    );
  }
  return formatMessage(
    '{user} streams {firstGame}, {secondGame} and {thirdGame}.',
    { firstGame, secondGame, thirdGame, user: displayName },
    'ChannelDescription',
  );
};

export const ChannelDescriptionBase: FC<ChannelDescriptionProps> = ({
  expanded,
  user,
}) => {
  const intl = useIntl();

  return (
    <CoreText color={Color.Alt} lines={expanded ? undefined : 2}>
      {getFormattedBio(intl, user)}
    </CoreText>
  );
};

ChannelDescriptionBase.displayName = 'ChannelDescription';

export const ChannelDescription = createFragmentContainer(
  ChannelDescriptionBase,
  {
    user: graphql`
      fragment ChannelDescription_user on User {
        login
        displayName
        description
        lastBroadcast {
          game {
            displayName
          }
        }
        videos(first: 30) {
          edges {
            node {
              id
              game {
                id
                displayName
              }
            }
          }
        }
      }
    `,
  },
);
