import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { StreamType, getStreamType, strongText } from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import { Color, CoreText, FontSize } from 'twitch-core-ui';
import type { TwitchIntl } from 'twitch-intl';
import type { ChannelStatus_user } from './__generated__/ChannelStatus_user.graphql';

type ChannelStatusProps = {
  user: ChannelStatus_user;
};

const getFormattedSummary = (
  { formatMessage, formatPastRelativeDate }: TwitchIntl,
  user: ChannelStatus_user,
) => {
  const streamType = getStreamType({
    broadcaster: user,
    type: user.stream?.type ?? null,
  });

  switch (streamType) {
    case StreamType.Live:
    case StreamType.Premiere:
      const viewersCountLabel = formatMessage(
        '{viewersCount, plural, one {# viewer} other {# viewers}}',
        {
          viewersCount: user.stream?.viewersCount ?? 0,
        },
        'ChannelProfileInfo',
      );

      return formatMessage(
        'Streaming <x:strong>{category}</x:strong> with <x:strong>{viewersCountLabel}</x:strong>',
        {
          category: user.stream?.game?.displayName ?? '',
          viewersCountLabel,
          'x:strong': strongText,
        },
        'ChannelProfileInfo',
      );
    case StreamType.Hosting:
      const hosting = user.hosting as NonNullable<
        ChannelStatus_user['hosting']
      >;

      return formatMessage(
        'Hosting <x:strong>{hostingChannel}</x:strong>',
        {
          hostingChannel: getFormattedUserDisplayName(hosting),
          'x:strong': strongText,
        },
        'ChannelProfileInfo',
      );
    case StreamType.Rerun:
      return user.stream?.createdAt
        ? formatMessage(
            'Streaming a rerun from {rerunCreatedAt}',
            {
              rerunCreatedAt: formatPastRelativeDate(
                new Date(user.stream.createdAt),
              ),
            },
            'ChannelProfileInfo',
          )
        : formatMessage('Streaming a rerun', 'ChannelProfileInfo');
    default:
      if (user.lastBroadcast?.startedAt) {
        const lastLiveDate = formatPastRelativeDate(
          new Date(user.lastBroadcast.startedAt),
        );

        return formatMessage(
          'Last live {lastLiveDate}',
          { lastLiveDate },
          'ChannelProfileInfo',
        );
      }
  }
};

export const ChannelStatusBase: FC<ChannelStatusProps> = ({ user }) => {
  const intl = useIntl();

  return (
    <CoreText color={Color.Base} ellipsis fontSize={FontSize.Size6}>
      {getFormattedSummary(intl, user)}
    </CoreText>
  );
};

ChannelStatusBase.displayName = 'ChannelStatus';

export const ChannelStatus = createFragmentContainer(ChannelStatusBase, {
  user: graphql`
    fragment ChannelStatus_user on User {
      hosting {
        id
        login
        displayName
      }
      lastBroadcast {
        id
        startedAt
        game {
          id
          displayName
        }
      }
      stream {
        id
        createdAt
        game {
          id
          displayName
        }
        type
        viewersCount
      }
    }
  `,
});
