import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import {
  AlignItems,
  Color,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  FlexWrap,
  FontSize,
  Layout,
  WordBreak,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../../routing';
import { SocialIcon } from './SocialIcon';
import type { SocialMediaLinks_user } from './__generated__/SocialMediaLinks_user.graphql';

/* In backend, the max length of a string should only be 50 characters. If strings were created before this limit existed,
this enforces that same behavior in the frontend.*/
const MAX_TEXT_LENGTH = 50;
/* If the length of a social media link is longer than 20 characters, introduces an arbitrary word break 20 characters in the string to wrap around nicely in the container*/
const BREAK_LINK_LIMIT = 10;

function formatLinkLength(link: string) {
  return link.substring(0, MAX_TEXT_LENGTH);
}

export type SocialMediaLinksProps = {
  expanded?: boolean;
  user: SocialMediaLinks_user;
};

const getLinks = (user: SocialMediaLinks_user, expanded?: boolean) => {
  const socialMedias = user.channel?.socialMedias;

  if (!socialMedias) {
    return null;
  }

  return socialMedias.slice(0, expanded ? 5 : 1);
};

export const SocialMediaLinksBase: FC<SocialMediaLinksProps> = ({
  expanded,
  user,
}) => {
  const links = getLinks(user, expanded);

  if (!links) {
    return null;
  }

  return (
    <Layout color={Color.Alt} display={Display.Flex} flexWrap={FlexWrap.Wrap}>
      {links.map((link) => {
        const { id, name, title, url } = link;
        const shouldBreakMediaLink = title.length > BREAK_LINK_LIMIT;

        return (
          <CoreLink
            hoverUnderlineNone
            key={id}
            linkTo="/deferToRenderLink"
            renderLink={renderTachyonLink({
              route: RouteName.External,
              routeParams: { href: url },
            })}
            role="link"
            targetBlank
            underline={false}
            variant={CoreLinkType.Inherit}
          >
            <Layout
              alignItems={AlignItems.Center}
              display={Display.Flex}
              flexWrap={FlexWrap.NoWrap}
              fullWidth
              padding={{ bottom: 0.5, right: 1 }}
            >
              <Layout
                alignItems={AlignItems.Center}
                display={Display.Flex}
                padding={{ right: 0.5 }}
              >
                <SocialIcon name={(name ?? '').toLowerCase()} />
              </Layout>

              <CoreText
                ellipsis
                fontSize={FontSize.Size7}
                wordBreak={
                  shouldBreakMediaLink ? WordBreak.BreakWord : undefined
                }
              >
                {formatLinkLength(title)}
              </CoreText>
            </Layout>
          </CoreLink>
        );
      })}
    </Layout>
  );
};

SocialMediaLinksBase.displayName = 'SocialMediaLinks';

export const SocialMediaLinks = createFragmentContainer(SocialMediaLinksBase, {
  user: graphql`
    fragment SocialMediaLinks_user on User {
      channel {
        id
        socialMedias {
          id
          name
          title
          url
        }
      }
    }
  `,
});
