import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { isAndroidOS, isIOS, useStaticEnvironment } from 'tachyon-environment';
import {
  AlignItems,
  CoreLink,
  Display,
  FlexWrap,
  JustifyContent,
  Layout,
  Overflow,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { ChannelAvatar, FollowButtonUpsell } from '../../../common';
import { ChannelDescription } from './ChannelDescription';
import { ChannelName } from './ChannelName';
import { ChannelStatus } from './ChannelStatus';
import { FollowerCount } from './FollowerCount';
import { SocialMediaLinks } from './SocialMediaLinks';
import type { ChannelProfileInfo_user } from './__generated__/ChannelProfileInfo_user.graphql';

type ChannelProfileInfoProps = {
  expanded?: boolean;
  user: ChannelProfileInfo_user;
};

const ScAvatarWrapper = styled(Layout)`
  margin-top: -1.5rem;
`;

export const ChannelProfileInfoBase: FC<ChannelProfileInfoProps> = ({
  expanded,
  user,
}) => {
  const { client } = useStaticEnvironment();
  const isAndroidOrIos =
    isAndroidOS(client.agentInfo) || isIOS(client.agentInfo);

  const followerCount = user.followers?.totalCount ?? null;

  return (
    <Layout
      alignItems={AlignItems.Start}
      display={Display.Flex}
      flexWrap={FlexWrap.NoWrap}
    >
      <ScAvatarWrapper flexGrow={0} flexShrink={0}>
        <CoreLink
          linkTo="/deferToRenderLink"
          renderLink={renderTachyonLink({
            route: RouteName.Channel,
            routeParams: { login: user.login },
          })}
        >
          <ChannelAvatar
            isLive={!!user.stream}
            login={user.login}
            primaryColorHex={user.primaryColorHex}
            profileImageURL={user.profileImageURL}
          />
        </CoreLink>
        {followerCount !== null && <FollowerCount count={followerCount} />}
      </ScAvatarWrapper>
      <Layout
        flexGrow={1}
        flexShrink={1}
        overflow={Overflow.Hidden}
        padding={{ left: 1 }}
      >
        <Layout
          display={Display.Flex}
          justifyContent={JustifyContent.Between}
          padding={{ y: 0.5 }}
        >
          <ChannelName user={user} />
          {isAndroidOrIos && (
            <Layout margin={{ left: 1 }}>
              <FollowButtonUpsell displayName={user.displayName} />
            </Layout>
          )}
        </Layout>
        <ChannelStatus user={user} />
        <Layout padding={{ top: 1 }}>
          <ChannelDescription expanded={expanded} user={user} />
        </Layout>
        <Layout padding={{ top: 1 }}>
          <SocialMediaLinks user={user} />
        </Layout>
      </Layout>
    </Layout>
  );
};

ChannelProfileInfoBase.displayName = 'ChannelProfileInfo';

export const ChannelProfileInfo = createFragmentContainer(
  ChannelProfileInfoBase,
  {
    user: graphql`
      fragment ChannelProfileInfo_user on User {
        ...ChannelStatus_user
        ...ChannelDescription_user
        ...ChannelName_user
        ...SocialMediaLinks_user
        profileImageURL(width: 150)
        login
        displayName
        primaryColorHex
        followers {
          totalCount
        }
        stream {
          id
        }
      }
    `,
  },
);
