import type { FC } from 'react';
import { useState } from 'react';
import type { GraphQLTaggedNode, VariablesOf } from 'react-relay/hooks';
import { graphql, useLazyLoadQuery } from 'react-relay/hooks';
import type { OperationType } from 'relay-runtime';
import { channelIsFound } from 'tachyon-page-utils';
import { Color, Layout } from 'twitch-core-ui';
import { NotFoundErrorLayout } from '../NotFoundErrorLayout';
import { Page } from '../Page';
import { ChannelCover } from './ChannelCover';
import { ChannelNav } from './ChannelNav';
import { ChannelProfileInfo } from './ChannelProfileInfo';
import type { ChannelLayout_Query } from './__generated__/ChannelLayout_Query.graphql';
import type { ChannelLayout_user$key } from './__generated__/ChannelLayout_user.graphql';

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelLayout_user on User {
    ...ChannelCover_user
    ...ChannelProfileInfo_user
    id
    login
  }
`;

type ChannelLayoutProps = {
  loading: boolean;
  login: string;
  // Enforces that layout data requirements are fetched
  // Actual data comes from the call to useQueryRelayCache
  user: ChannelLayout_user$key | null;
};

// Experimental
//
// Not added to tachyon-relay until this pattern is trialed / fleshed out.
// We probably want the signature that passes the parent ref in so that we can typecheck that a higher up parent has fetched the data needs.
function useQueryRelayCache<Query extends OperationType>(
  query: GraphQLTaggedNode,
  variables: VariablesOf<Query>,
): Query['response'] {
  return useLazyLoadQuery<Query>(query, variables, {
    fetchPolicy: 'store-only',
  });
}

export const ChannelLayout: FC<ChannelLayoutProps> = ({
  children,
  loading,
  login,
}) => {
  /*
   * Fetch the user data that powers the channel chrome from the relay cache.
   * This lets us preserve the chrome across channel page navigations. Otherwise,
   * the chrome is blown away by the next channel page's query even though the
   * channel chrome data is identical across the page queries.
   *
   * This implicitly relies on complete overlap with the channel page queries.
   * Relay can resolve this from the cache, so there is no addtional fetchQuery needed to populate the store.
   */
  const { user } = useQueryRelayCache<ChannelLayout_Query>(
    graphql`
      query ChannelLayout_Query($login: String!) {
        user(login: $login) {
          ...ChannelLayout_user @relay(mask: false)
        }
      }
    `,
    { login },
  );

  /*
   * TODO: Wire this up based on a user action.
   * VX: https://jira.xarth.tv/browse/VXCHAN-415
   *
   * This needs to be user initiated. Doing it automatically on the about page causes the entire page layout to shift
   * and looks like a bug.
   *
   */
  // eslint-disable-next-line tachyon/prefer-use-const
  const [expanded] = useState(false);

  if (!channelIsFound(user)) {
    if (loading) {
      return null;
    }
    return <NotFoundErrorLayout />;
  }

  return (
    <Page>
      <ChannelCover user={user} />
      <Layout padding={{ bottom: 0.5, top: 0.5, x: 1 }}>
        <ChannelProfileInfo expanded={expanded} user={user} />
      </Layout>
      <Layout padding={{ top: 0.5, x: 2 }}>
        <Layout color={Color.Base} padding={{ bottom: 1 }}>
          <ChannelNav login={user.login} />
        </Layout>
        {children}
      </Layout>
    </Page>
  );
};

ChannelLayout.displayName = 'ChannelLayout';
