import type { FC } from 'react';
import styled, { keyframes } from 'styled-components';
import { InteractionType, useInteractionTracking } from 'tachyon-event-tracker';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Background,
  BorderRadius,
  Button,
  ButtonType,
  Color,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  JustifyContent,
  Layout,
  OverlayRegion,
  Position,
  Title,
  TitleSize,
} from 'twitch-core-ui';
import { useTachyonRouter } from '../../../../routing';

const translateUpKeyframes = keyframes`
0%{
  transform: translateY(150%);
}
100%{
  transform: translateY(0%);
}
`;

type ScStatusTranslateUpCardProps = {
  show: boolean;
};

/* for a11y, visibility delayed in transition by the same time the element takes to animate */
const ANIMATION_DURATION = 300;

export const ScStatusTranslateUpCard = styled(
  Layout,
)<ScStatusTranslateUpCardProps>`
  animation: ${translateUpKeyframes} ${ANIMATION_DURATION}ms ease 1;
  transform: ${(props: ScStatusTranslateUpCardProps): string =>
    !props.show ? 'translateY(150%)' : ''};
  transition: visibility 0s linear ${ANIMATION_DURATION}ms,
    transform ${ANIMATION_DURATION}ms ease;
  visibility: ${(props: ScStatusTranslateUpCardProps): string =>
    !props.show ? 'hidden' : 'visible'};
`;

type MessageCardProps = {
  show: boolean;
};

export const RELOAD_INTERACTION_VALUE = 'network_restored_reload';

export const OnlineMessageCard: FC<MessageCardProps> = ({ show }) => {
  const { formatMessage } = useIntl();
  const trackInteraction = useInteractionTracking();
  const { reload } = useTachyonRouter();
  return (
    <OverlayRegion>
      <ScStatusTranslateUpCard
        alignItems={AlignItems.Center}
        as="nav"
        attachBottom
        attachLeft
        attachRight
        background={Background.Accent}
        borderRadius={BorderRadius.Medium}
        color={Color.Base}
        display={Display.Flex}
        elevation={2}
        flexDirection={FlexDirection.Row}
        justifyContent={JustifyContent.Between}
        margin={1}
        padding={{ x: 2, y: 1 }}
        position={Position.Absolute}
        show={show}
      >
        <Layout display={Display.Flex} flexDirection={FlexDirection.Column}>
          <Title size={TitleSize.Small}>
            {formatMessage('Praise Helix!', 'OfflinePageWrapper')}
          </Title>
          <CoreText fontSize={FontSize.Size6}>
            {formatMessage(
              'It looks like your internet is back!',
              'OfflinePageWrapper',
            )}
          </CoreText>
        </Layout>
        <Button
          onClick={() => {
            trackInteraction({
              interaction: InteractionType.Click,
              interactionContent: RELOAD_INTERACTION_VALUE,
            });
            reload();
          }}
          overlay
          variant={ButtonType.Primary}
        >
          {formatMessage('Reload', 'OfflinePageWrapper')}
        </Button>
      </ScStatusTranslateUpCard>
    </OverlayRegion>
  );
};

OnlineMessageCard.displayName = 'OnlineMessageCard';

// istanbul ignore next: trivial
export const OfflineMessageCard: FC<MessageCardProps> = ({ show }) => {
  const { formatMessage } = useIntl();
  return (
    <OverlayRegion>
      <ScStatusTranslateUpCard
        alignItems={AlignItems.Center}
        as="aside"
        attachBottom
        attachLeft
        attachRight
        background={Background.Base}
        borderRadius={BorderRadius.Medium}
        color={Color.Base}
        display={Display.Flex}
        elevation={2}
        flexDirection={FlexDirection.Row}
        justifyContent={JustifyContent.Between}
        margin={1}
        padding={{ x: 2, y: 1 }}
        position={Position.Absolute}
        show={show}
      >
        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          padding={{ y: 0.5 }}
        >
          <CoreText bold fontSize={FontSize.Size6}>
            {formatMessage(
              "Looks like your internet isn't working...",
              'NetworkNotification',
            )}
          </CoreText>
        </Layout>
      </ScStatusTranslateUpCard>
    </OverlayRegion>
  );
};

OfflineMessageCard.displayName = 'OfflineMessageCard';
