import type { FC } from 'react';
import { useDynamicEnvironment } from 'tachyon-environment';
import { usePrevious } from 'tachyon-utils';
import { Transition, TransitionType } from 'twitch-core-ui';
import { OfflineMessageCard, OnlineMessageCard } from './MessageCards';

type NetworkNotificationProps = {
  enable: boolean;
  /**
   * Whether or not an offline page is showing. This controls whether to show the OnlineMessageCard or the OfflineMessageCard.
   *
   * The OfflineMessageCard will only show if isOffline: false
   * The OnlineMessageCard will only show if isOffline: true
   */
  isOfflinePage: boolean;
};

export const NetworkNotification: FC<NetworkNotificationProps> = ({
  children,
  enable,
  isOfflinePage,
}) => {
  /*
   * If the navigator.onLine is false, we always show the OfflineMessageCard, because we know the user cannot connect to the internet.
   * The reverse is not necessarily true: navigator.onLine true may not indicate that the network is working. The user could be connected to a router, and that router could not have internet connection. If the user transitioned from onLine false to onLine true, then we can reasonably assume a connection error has been fixed, and show the OnlineMessageCard.
   */
  const { isOffline } = useDynamicEnvironment();
  const wasOffline = usePrevious(isOffline);
  const showOffline = enable && isOffline && !isOfflinePage;
  const showOnline = enable && wasOffline && !isOffline && isOfflinePage;

  return (
    <>
      {children}
      <Transition injectChild show={showOnline} type={TransitionType.Fade}>
        <OnlineMessageCard show={showOnline} />
      </Transition>
      <Transition injectChild show={showOffline} type={TransitionType.Fade}>
        <OfflineMessageCard show={showOffline} />
      </Transition>
    </>
  );
};

NetworkNotification.displayName = 'NetworkNotification';
