import { useDynamicEnvironment } from 'tachyon-environment';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import { OfflineMessageCard, OnlineMessageCard } from './MessageCards';
import { NetworkNotification } from '.';

jest.mock('tachyon-environment', () => ({
  ...jest.requireActual('tachyon-environment'),
  useDynamicEnvironment: jest.fn(),
}));

const mockUseDynamicEnvironment = useDynamicEnvironment as jest.Mock;

describe(NetworkNotification, () => {
  const setup = createShallowWrapperFactory(NetworkNotification, () => ({
    enable: true,
    isOfflinePage: true,
  }));

  describe('when enable is false', () => {
    it('does not render OfflineMessageCard', () => {
      mockUseDynamicEnvironment.mockImplementationOnce(() => ({
        isOffline: true,
      }));
      const { wrapper } = setup({ enable: false, isOfflinePage: false });

      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: false });
      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: false });
    });

    it('renders OnlineMessageCard when transitioning offline => online', () => {
      mockUseDynamicEnvironment
        .mockImplementationOnce(() => ({ isOffline: true }))
        .mockImplementationOnce(() => ({ isOffline: false }));

      const { props, wrapper } = setup({ enable: false, isOfflinePage: true });
      // force rerender to simulate environment updated
      wrapper.setProps(props);

      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: false });
      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: false });
    });
  });

  describe('when on an online page', () => {
    it('renders OfflineMessageCard when navigator is offline', () => {
      mockUseDynamicEnvironment.mockImplementationOnce(() => ({
        isOffline: true,
      }));
      const { wrapper } = setup({ isOfflinePage: false });

      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: true });
      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: false });
    });

    it('does not render OnlineMessageCard when transitioning offline => online', () => {
      mockUseDynamicEnvironment
        .mockImplementationOnce(() => ({ isOffline: true }))
        .mockImplementationOnce(() => ({ isOffline: false }));

      const { props, wrapper } = setup({ isOfflinePage: false });
      // force rerender to simulate environment updated
      wrapper.setProps(props);

      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: false });
      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: false });
    });
  });

  describe('when on an offline page', () => {
    it('does not render OfflineMessageCard when navigator is offline', () => {
      mockUseDynamicEnvironment.mockImplementationOnce(() => ({
        isOffline: false,
      }));
      const { wrapper } = setup({ isOfflinePage: true });

      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: false });
      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: false });
    });

    it('renders OnlineMessageCard when transitioning offline => online', () => {
      mockUseDynamicEnvironment
        .mockImplementationOnce(() => ({ isOffline: true }))
        .mockImplementationOnce(() => ({ isOffline: false }));

      const { props, wrapper } = setup({ isOfflinePage: true });
      // force rerender to simulate environment updated
      wrapper.setProps(props);

      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: true });
      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: false });
    });

    it('does not render OnlineMessageCard when online without transitioning from offline', () => {
      mockUseDynamicEnvironment.mockImplementationOnce(() => ({
        isOffline: false,
      }));

      const { wrapper } = setup({ isOfflinePage: true });

      expect(wrapper.find(OnlineMessageCard)).toHaveProp({ show: false });
      expect(wrapper.find(OfflineMessageCard)).toHaveProp({ show: false });
    });
  });
});
