import type { FC } from 'react';
import styled from 'styled-components';
import {
  safeAreaBottomPaddingToRemoveInsets,
  safeAreaHorizontalPaddingToRemoveInsets,
} from 'tachyon-more-ui';
import {
  AlignItems,
  Display,
  JustifyContent,
  Layout,
  LoadingSpinner,
  Position,
  SpinnerSize,
} from 'twitch-core-ui';
import { LOADING_SHOW_DELAY_MS, TOP_NAV_HEIGHT_PX } from '../../../config';
import { TopNav } from '../../common';

type PageProps = {
  hideTopNav?: boolean;
  loading?: boolean;
};

// For "-webkit-overflow-scrolling" (no-op for iOS 13 onward), see:
// https://developer.mozilla.org/en-US/docs/Web/CSS/-webkit-overflow-scrolling
//
// bottom padding only to prevent elements being forced under the iOS switcher
// "handle" in the unsafe area when in PWA mode
const ScMain = styled.main<Pick<PageProps, 'hideTopNav'>>`
  height: ${({ hideTopNav }) =>
    hideTopNav ? '100%' : `calc(100% - ${TOP_NAV_HEIGHT_PX}px)`};
  margin-top: ${({ hideTopNav }) => (hideTopNav ? 0 : TOP_NAV_HEIGHT_PX)}px;
  ${safeAreaBottomPaddingToRemoveInsets}
  ${safeAreaHorizontalPaddingToRemoveInsets}
  -webkit-overflow-scrolling: touch;
  overflow-y: scroll;
  width: 100%;
`;

/**
 * Tomorrow's Page layout component. The component is absolutely positioned and
 * attached to the screen's full height and width since "100vh" on a mobile device
 * includes the visible site height and the height of the browser's native UI URL bar
 * which results in partial overflow for flex children that use height 100% for sizing.
 *
 * Because the layout around "main" is absolutely positioned, "main" has vertical overflow
 * scrolling.
 */
export const Page: FC<PageProps> = ({ children, hideTopNav, loading }) => (
  <Layout
    attachBottom
    attachLeft
    attachRight
    attachTop
    position={Position.Absolute}
  >
    {!hideTopNav && <TopNav />}
    <ScMain hideTopNav={hideTopNav}>
      {loading ? (
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          fullHeight
          fullWidth
          justifyContent={JustifyContent.Center}
        >
          <LoadingSpinner
            delay={LOADING_SHOW_DELAY_MS}
            size={SpinnerSize.Large}
          />
        </Layout>
      ) : (
        children
      )}
    </ScMain>
  </Layout>
);

Page.displayName = 'Page';
