import { PlayerControllerRoot } from 'pulsar';
import type { FC, ReactNode } from 'react';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Background,
  Color,
  CoreText,
  Display,
  FontSize,
  JustifyContent,
  Layout,
} from 'twitch-core-ui';
import { StaticTokenMap } from 'twitch-core-ui-tokens';

export type PlayerFrameProps = {
  player: ReactNode;
};

export const LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY =
  'only screen and (min-device-width: 768px) and (orientation: landscape)';

export const LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY =
  'only screen and (max-device-width: 768px) and (orientation: landscape)';

export const LARGE_SCREEN_LANDSCAPE_PANEL_WIDTH_PX = '320';
export const SMALL_SCREEN_LANDSCAPE_PANEL_WIDTH_PX = '230';

/**
 * Ensure that the player size is exactly 16:9 in portrait mode.
 */
export const PORTRAIT_PLAYER_HEIGHT = 'calc(100vw * 9 / 16)';

/**
 * Add padding below the controls to get them in frame on landscape tablets
 */
export const BELOW_CONTROLS_PADDING_PX = 30;

const ScPlayerLayout = styled.div`
  display: flex;
  flex-direction: column;
  height: 100%;

  @media ${LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY} {
    overflow: scroll;
  }

  @media ${LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY} {
    flex-direction: row;
  }
`;

const ScPlayerContainer = styled.section`
  background: ${StaticTokenMap['color-black']};
  flex-shrink: 0;
  height: ${PORTRAIT_PLAYER_HEIGHT};

  @media ${LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY} {
    height: 100%;
  }

  @media ${LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY} {
    height: 100%;
    padding-bottom: ${BELOW_CONTROLS_PADDING_PX}px;
    width: calc(100vw - ${LARGE_SCREEN_LANDSCAPE_PANEL_WIDTH_PX}px);
  }
`;

const ScSecondaryContent = styled(Layout)`
  height: calc(100% - ${PORTRAIT_PLAYER_HEIGHT});

  @media ${LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY} {
    height: 100%;
  }

  @media ${LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY} {
    height: 100%;
    width: ${LARGE_SCREEN_LANDSCAPE_PANEL_WIDTH_PX}px;
  }
`;

const PulsarErrorMessage: FC = () => {
  const { formatMessage } = useIntl();

  return (
    <Layout
      alignItems={AlignItems.Center}
      display={Display.Flex}
      fullHeight
      fullWidth
      justifyContent={JustifyContent.Center}
    >
      <CoreText color={Color.Overlay} fontSize={FontSize.Size3}>
        {formatMessage(
          'An error occurred. Please refresh the page.',
          'PulsarUniversalPlayer',
        )}
      </CoreText>
    </Layout>
  );
};
PulsarErrorMessage.displayName = 'PulsarErrorMessage';

// istanbul ignore next: trivial
/**
 * Provides a layout containing a player and additional content that
 * is optimally presented for various screen widths and orientations.
 */
export const PlayerLayout: FC<PlayerFrameProps> = ({ children, player }) => {
  return (
    <PlayerControllerRoot>
      <ScPlayerLayout>
        <ScPlayerContainer>{player}</ScPlayerContainer>
        <ScSecondaryContent as="section" background={Background.Base}>
          {children}
        </ScSecondaryContent>
      </ScPlayerLayout>
    </PlayerControllerRoot>
  );
};

PlayerLayout.displayName = 'PlayerFrame';
