import { useRouter } from 'next/router';
import { defaultPageviewTracking } from 'tachyon-page-utils';
import {
  flattenHeaderOrParam,
  isBrowser,
  useConst,
  useEffectOnce,
} from 'tachyon-utils';
import { APP_SHELL_REDIRECT_QUERY_KEY } from '../../../config';
import {
  RouteName,
  routeNameFromPathname,
  routeNameFromRawPath,
} from '../../../routing';
import type { TomorrowPage } from '../types';

/**
 * This page acts a minimal "App Shell" that is cached by the service worker and served as an offline fallback when the initial page request fails. See internal-docs/apps/tomorrow/progressive-web-application.md for more context.
 *
 * All initial page requests that fail fallback to this page ("/", "/directory", etc). We redirect client side to the requested pathname so that the standard next.js lifecycle is triggered.
 *
 * To test this flow, disable your network connection and request any mweb page.
 */
export const AppShell: TomorrowPage = () => {
  const router = useRouter();
  // We specifically use the window.location.pathname here instead of the
  // router's pathname. When this page was cached on service worker
  // installation, the router's pathname was `/app-shell`. When we fallback to
  // this page in the offline case we render this cached page in place, and then
  // redirect to the requested page.
  //
  // We need to capture the search here before the RouterUtilsRoot strips
  // them out. If we performed this search in useEffect the RouterUtilsRoot
  // will have already stripped the search.
  const path = useConst(() => {
    if (!isBrowser()) {
      // we won't execute this path when we can actually redirect in the browser,
      // but default to '/' for safety anyway
      return '/';
    }

    const encodedRedirectPath = flattenHeaderOrParam(
      router.query[APP_SHELL_REDIRECT_QUERY_KEY],
    );
    if (encodedRedirectPath) {
      const redirectPath = decodeURIComponent(encodedRedirectPath);
      // prevent attempting to navigate to invalid routes
      return routeNameFromRawPath(redirectPath) !== RouteName.NotFound
        ? redirectPath
        : '/';
    }

    const swIntendedPathname =
      // This prevents a redirect loop when directly navigating to /app-shell
      routeNameFromPathname(window.location.pathname) !== RouteName.AppShell
        ? window.location.pathname
        : '/';

    return swIntendedPathname + window.location.search;
  });

  useEffectOnce(() => {
    router.replace(path);
  });

  return null;
};

AppShell.displayName = 'AppShell';
AppShell.pageviewTracking = defaultPageviewTracking;
AppShell.requiresJsForInteractivity = true;
