import { createMountWrapperFactory } from 'tachyon-test-utils';
import { APP_SHELL_REDIRECT_QUERY_KEY } from '../../../config';
import { mockRouter } from '../../../routing/test-mocks';
import { AppShell } from '.';

let mockQuery = {};
jest.mock('next/router', () => ({
  useRouter: () => ({ ...mockRouter, query: mockQuery }),
}));

describe(AppShell, () => {
  const ogLocation = window.location;
  beforeAll(() => {
    // @ts-expect-error: tests
    delete window.location;
  });

  afterEach(() => {
    mockQuery = {};
    window.location = ogLocation;
  });

  const setup = createMountWrapperFactory(AppShell);

  it('redirects to the homepage when navigated to directly', () => {
    window.location = {
      pathname: '/app-shell',
      search: '',
    } as typeof window.location;
    setup();

    expect(mockRouter.replace).toHaveBeenCalledWith('/');
  });

  it('redirects to the intended page when in AppShell app boot fallback', () => {
    window.location = {
      pathname: '/search',
      search: '?term=timthetatman',
    } as typeof window.location;
    setup();

    expect(mockRouter.replace).toHaveBeenCalledWith(
      '/search?term=timthetatman',
    );
  });

  describe('redirect query param', () => {
    type RedirectQueryParamTestCase = {
      description: string;
      path: string;
    };

    it.each`
      description                | path
      ${'channel page'}          | ${'/lirik'}
      ${'home page'}             | ${'/'}
      ${'multi-part path'}       | ${'/directory/game/Fortnite'}
      ${'path with query param'} | ${'/search?term=fortnite'}
    `(
      'redirects to the intended page for $description',
      ({ path }: RedirectQueryParamTestCase) => {
        mockQuery = {
          [APP_SHELL_REDIRECT_QUERY_KEY]: encodeURIComponent(path),
        };
        setup();

        expect(mockRouter.replace).toHaveBeenCalledWith(path);
      },
    );

    it.each`
      description                | path
      ${'multi-part path'}       | ${'/bad/actors/are/everywhere/out/there'}
      ${'path with query param'} | ${'/does/this/thing/work?pwn=true'}
    `(
      'redirects to the homepage for invalid route with $description',
      ({ path }: RedirectQueryParamTestCase) => {
        mockQuery = {
          [APP_SHELL_REDIRECT_QUERY_KEY]: encodeURIComponent(path),
        };
        setup();

        expect(mockRouter.replace).toHaveBeenCalledWith('/');
      },
    );
  });
});
