import { PlayerControllerRoot } from 'pulsar';
import type { FC, ReactNode } from 'react';
import { useState } from 'react';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import {
  ButtonIcon,
  Display,
  Layout,
  Position,
  SVGAsset,
} from 'twitch-core-ui';
import { StaticTokenMap } from 'twitch-core-ui-tokens';
import {
  BELOW_CONTROLS_PADDING_PX,
  LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY,
  LARGE_SCREEN_LANDSCAPE_PANEL_WIDTH_PX,
  PORTRAIT_PLAYER_HEIGHT,
  SMALL_SCREEN_LANDSCAPE_PANEL_WIDTH_PX,
} from '../../../layouts';

const ScPlayerLayout = styled(Layout)`
  flex-direction: column;

  @media only screen and (orientation: landscape) {
    flex-direction: row;
  }
`;

const ScPlayerContainer = styled(Layout)<{ $showRightColumn: boolean }>`
  background: ${StaticTokenMap['color-black']};
  height: ${PORTRAIT_PLAYER_HEIGHT};

  @media only screen and (orientation: landscape) {
    height: 100%;
    padding-left: ${({ $showRightColumn }) =>
      $showRightColumn ? '0' : '3rem'};
    padding-right: ${({ $showRightColumn }) =>
      $showRightColumn ? '0' : '3rem'};
  }

  @media ${LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY} {
    padding-bottom: ${BELOW_CONTROLS_PADDING_PX}px;
  }
`;

const ScRightColumnContainer = styled(Layout)`
  width: ${SMALL_SCREEN_LANDSCAPE_PANEL_WIDTH_PX}px;

  @media ${LANDSCAPE_WIDE_SCREEN_MEDIA_QUERY} {
    width: ${LARGE_SCREEN_LANDSCAPE_PANEL_WIDTH_PX}px;
  }
`;

const ScRightColumn = styled.div`
  @media (orientation: portrait) {
    display: none;
  }
`;

const ScSecondaryContent = styled(Layout)`
  height: calc(100% - ${PORTRAIT_PLAYER_HEIGHT});

  @media only screen and (orientation: landscape) {
    display: none;
  }
`;

export const ChannelPlayerLayout: FC<{ player: ReactNode }> = ({
  children,
  player,
}) => {
  const { formatMessage } = useIntl();
  const [showRightColumn, setShowRightColumn] = useState(true);

  return (
    <PlayerControllerRoot>
      <ScPlayerLayout
        display={Display.Flex}
        fullHeight
        fullWidth
        position={Position.Relative}
      >
        <ScPlayerContainer $showRightColumn={showRightColumn} flexGrow={1}>
          {player}
        </ScPlayerContainer>
        {/*
          the right column and secondary content are both needed since the
          right column only appears in landscape and secondary content only
          appears in portrait
        */}
        <ScRightColumn>
          {!showRightColumn && (
            <Layout
              attachRight
              attachTop
              padding={0.5}
              position={Position.Absolute}
            >
              <ButtonIcon
                aria-label={formatMessage('Show Chat', 'ChannelPage')}
                icon={SVGAsset.ColSlideLeft}
                onClick={() => {
                  setShowRightColumn((prev) => !prev);
                }}
                overlay
              />
            </Layout>
          )}
          <ScRightColumnContainer
            display={showRightColumn ? undefined : Display.Hide}
            fullHeight
            position={Position.Relative}
          >
            <Layout
              attachLeft
              attachTop
              padding={0.5}
              position={Position.Absolute}
            >
              <ButtonIcon
                aria-label={formatMessage('Hide Chat', 'ChannelPage')}
                icon={SVGAsset.ColSlideRight}
                onClick={() => {
                  setShowRightColumn((prev) => !prev);
                }}
              />
            </Layout>
            {children}
          </ScRightColumnContainer>
        </ScRightColumn>
        <ScSecondaryContent>{children}</ScSecondaryContent>
      </ScPlayerLayout>
    </PlayerControllerRoot>
  );
};
ChannelPlayerLayout.displayName = 'ChannelPlayerLayout';
