import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { TagPage } from 'tachyon-discovery';
import { isAndroidOS, isIOS, useStaticEnvironment } from 'tachyon-environment';
import { useIntl } from 'tachyon-intl';
import { isValidObject } from 'tachyon-relay';
import type { OmitRefType } from 'tachyon-type-library';
import {
  AlignItems,
  AlignSelf,
  Color,
  CoreLink,
  CoreLinkType,
  CoreText,
  Display,
  FlexDirection,
  FontSize,
  Layout,
  LineHeight,
  Overflow,
  Tooltip,
  WhiteSpace,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import {
  ContentInfoBoxContainer,
  FollowButtonUpsell,
  InfoBoxAvatar,
  StreamTagList,
  WebShareButton,
  canShare,
} from '../../../common';
import type { StreamInfoBox_channel } from './__generated__/StreamInfoBox_channel.graphql';
import type { StreamInfoBox_channelBase } from './__generated__/StreamInfoBox_channelBase.graphql';

export type StreamInfoBoxProps = {
  channel: StreamInfoBox_channel;
};

export const StreamInfoBoxBase: FC<StreamInfoBoxProps> = ({ channel }) => {
  const { formatMessage } = useIntl();
  const { client } = useStaticEnvironment();

  const isAndroidOrIos =
    isAndroidOS(client.agentInfo) || isIOS(client.agentInfo);

  // TODO: convert hosting status dependent data to use `getPlayingChannel`
  const {
    isHosted,
    playingChannel,
  }: {
    isHosted: boolean;
    playingChannel: OmitRefType<StreamInfoBox_channelBase>;
  } =
    isValidObject(channel.hosting) && isValidObject(channel.hosting.stream)
      ? { isHosted: true, playingChannel: channel.hosting }
      : { isHosted: false, playingChannel: channel };

  const playingStream = playingChannel.stream;
  const streamGameName = playingStream?.game?.name ?? null;

  return (
    <ContentInfoBoxContainer>
      <Layout
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        fullWidth
      >
        <Layout
          alignItems={AlignItems.Center}
          display={Display.Flex}
          flexDirection={FlexDirection.Row}
        >
          <Layout
            alignSelf={AlignSelf.Center}
            display={Display.Flex}
            padding={1}
          >
            <InfoBoxAvatar owner={playingChannel} />
          </Layout>
          <Layout
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            flexGrow={1}
            overflow={Overflow.Auto}
          >
            <Tooltip label={channel.displayName || channel.login}>
              <CoreText
                color={Color.Base}
                ellipsis
                fontSize={FontSize.Size3}
                lineHeight={LineHeight.Heading}
                whiteSpace={WhiteSpace.NoWrap}
              >
                {channel.displayName || channel.login}
              </CoreText>
            </Tooltip>
            <Layout overflow={Overflow.Auto}>
              {isHosted && (
                <CoreText
                  color={Color.Alt}
                  fontSize={FontSize.Size5}
                  whiteSpace={WhiteSpace.NoWrap}
                >
                  {formatMessage(
                    'hosting {hostedChannel}',
                    {
                      hostedChannel: (
                        <CoreLink
                          linkTo="/deferToRenderLink"
                          renderLink={renderTachyonLink({
                            route: RouteName.Channel,
                            routeParams: { login: playingChannel.login },
                          })}
                          variant={CoreLinkType.Default}
                        >
                          {playingChannel.displayName}
                        </CoreLink>
                      ),
                    },
                    'InfoBoxHosting',
                  )}
                </CoreText>
              )}
              {streamGameName && (
                <CoreText
                  color={Color.Alt}
                  fontSize={FontSize.Size5}
                  whiteSpace={WhiteSpace.NoWrap}
                >
                  {formatMessage(
                    'Playing {game}',
                    {
                      game: (
                        <CoreLink
                          linkTo="/deferToRenderLink"
                          renderLink={renderTachyonLink({
                            route: RouteName.GameDirectory,
                            routeParams: { gameAlias: streamGameName },
                          })}
                          variant={CoreLinkType.Default}
                        >
                          {streamGameName}
                        </CoreLink>
                      ),
                    },
                    'InfoBoxPlaying',
                  )}
                </CoreText>
              )}
            </Layout>
          </Layout>

          {isAndroidOrIos && (
            <Layout padding={{ x: 1, y: 1 }}>
              <FollowButtonUpsell displayName={channel.displayName} />
            </Layout>
          )}
          {canShare() && (
            <Layout padding={{ right: 1, y: 1 }}>
              <WebShareButton iconOnly />
            </Layout>
          )}
        </Layout>
        {playingStream && (
          <Layout overflow={Overflow.Scroll} padding={{ x: 1 }}>
            <StreamTagList
              gameAlias={streamGameName}
              page={TagPage.Stream}
              stream={playingStream}
            />
          </Layout>
        )}
      </Layout>
    </ContentInfoBoxContainer>
  );
};

StreamInfoBoxBase.displayName = 'StreamInfoBoxBase';

// Ensures that the same data requirements are met for both the parent and hosted channel
// eslint-disable-next-line no-unused-expressions
graphql`
  fragment StreamInfoBox_channelBase on User {
    id
    login
    displayName
    profileImageURL(width: 50)
    stream {
      id
      game {
        name
      }
      ...TagList_stream
    }
  }
`;

export const StreamInfoBox = createFragmentContainer(StreamInfoBoxBase, {
  channel: graphql`
    fragment StreamInfoBox_channel on User {
      ...StreamInfoBox_channelBase @relay(mask: false)
      hosting {
        ...StreamInfoBox_channelBase @relay(mask: false)
      }
    }
  `,
});
