import { graphql } from 'react-relay/hooks';
import styled from 'styled-components';
import type { ChatWorkerInstance } from 'tachyon-chat';
import { ChatRoot } from 'tachyon-chat-ui';
import { useIntl } from 'tachyon-intl';
import type {
  ChannelInitialProps,
  ChannelPathParameters,
} from 'tachyon-page-utils';
import {
  channelIsFound,
  channelIsNotFoundServerside,
  channelPathGetInitialProps,
  channelWatchPageviewTracking,
  getPlayingChannel,
} from 'tachyon-page-utils';
import { convertToUnsafeID } from 'tachyon-relay';
import { getFormattedUserDisplayName, isBrowser } from 'tachyon-utils';
import {
  Color,
  CoreText,
  Display,
  FlexDirection,
  Layout,
  TextAlign,
  TextTransform,
} from 'twitch-core-ui';
import { CLIENT_ID } from '../../../config';
import { Redirect, RouteName } from '../../../routing';
// @ts-expect-error: WebWorkers are necessarily (untyped) JS
// eslint-disable-next-line import/no-internal-modules
import ChatWorker from '../../../workers/chat.worker';
import { PulsarStreamPlayer } from '../../Player';
import { ChatPane } from '../../common';
import type { PageHeadQueryVariables } from '../../growth';
import {
  BranchEvent,
  TachyonBranchTimedTracker,
  pageHeadQueryVariables,
} from '../../growth';
import {
  LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY,
  NotFoundErrorLayout,
  Page,
} from '../../layouts';
import { ChannelHomePage } from '../ChannelHomePage';
import type { TomorrowPage } from '../types';
import { ChannelPlayerLayout } from './ChannelPlayerLayout';
import { StreamInfoBox } from './StreamInfoBox';
import type { Channel_QueryResponse } from './__generated__/Channel_Query.graphql';

/**
 * The user is able to scroll back through chat history. Set this sufficiently
 * high that they get some history, but this doesn't become a performance concern
 * since the messages aren't virtualized.
 */
const CHAT_BUFFER_SIZE = 100;

const ScStreamInfoBoxContainer = styled(Layout)`
  @media ${LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY} {
    display: none;
  }
`;

const ScSecondaryContentHeader = styled(Layout)`
  display: none;
  @media ${LANDSCAPE_NARROW_SCREEN_MEDIA_QUERY} {
    display: inline;
  }
`;

type ChannelPageInitialProps = ChannelInitialProps<
  PageHeadQueryVariables & { isBrowser: boolean }
>;

export type ChannelProps = Channel_QueryResponse & ChannelPageInitialProps;

/**
 * We show the channel home for offline channels. We avoid a server redirect for SEO purposes (Google penalizes unstable redirects).
 *
 * For non-existent channels we want to instead show a 404.
 */
export function shouldRenderChannelHome(
  channel: Channel_QueryResponse['channel'],
): ReturnType<typeof channelIsFound> {
  return channelIsFound(channel) && !getPlayingChannel(channel);
}

export const Channel: TomorrowPage<
  ChannelPageInitialProps,
  ChannelProps,
  ChannelPathParameters
> = (props) => {
  const { channel, channelHome } = props;
  const { formatMessage } = useIntl();

  if (!channelIsFound(channel)) {
    return <NotFoundErrorLayout />;
  }

  // Handle client side navigations to `/:login` when the channel is offline.
  // This edge case can occur when the user follows stale channel links, or when the channel goes offline while the user is navigating to the page.
  const clientSideNavigation = props.queryVariables.isBrowser;
  if (clientSideNavigation && shouldRenderChannelHome(channel)) {
    return (
      <Redirect
        params={{
          route: RouteName.ChannelHome,
          routeParams: { login: channel.login },
        }}
        replace
      />
    );
  }

  const displayName = getFormattedUserDisplayName(channel);

  // channelHome will always be present when the initial fetch occured server-side, but because it is marked with a GQL skip directive the autogenerated GQL type is marked as optional.
  //
  // If this invariant is violated, it's because backend services are in a degraded state, so we fall back to trying to render the channel page.
  if (shouldRenderChannelHome(channel) && channelHome) {
    const channelHomePageProps = { ...props, channel: channelHome };
    return <ChannelHomePage {...channelHomePageProps} loading={false} />;
  }

  return (
    <Page>
      <ChannelPlayerLayout player={<PulsarStreamPlayer channel={channel} />}>
        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          fullHeight
        >
          <ScStreamInfoBoxContainer>
            <StreamInfoBox channel={channel} />
          </ScStreamInfoBoxContainer>
          <ScSecondaryContentHeader
            borderBottom
            fullWidth
            padding={{ y: 1 }}
            textAlign={TextAlign.Center}
          >
            <CoreText
              bold
              color={Color.Base}
              transform={TextTransform.Uppercase}
            >
              {formatMessage('Stream Chat', 'ChannelPage')}
            </CoreText>
          </ScSecondaryContentHeader>
          <ChatRoot
            bufferSize={CHAT_BUFFER_SIZE}
            channel={{
              id: convertToUnsafeID(channel.id),
              login: channel.login,
            }}
            clientApiId={CLIENT_ID}
            getWorker={() =>
              // eslint-disable-next-line @typescript-eslint/no-unsafe-call
              new ChatWorker() as ChatWorkerInstance
            }
          >
            <ChatPane channelDisplayName={displayName} />
          </ChatRoot>
          <TachyonBranchTimedTracker
            delaySecondsUntilTrack={5 * 60}
            event={BranchEvent.FiveMinutePlay}
          />
        </Layout>
      </ChannelPlayerLayout>
    </Page>
  );
};

Channel.displayName = 'ChannelPage';
Channel.totalCacheLife = 45;
Channel.staleWhileRevalidate = 30;
Channel.isNotFoundServerside = channelIsNotFoundServerside;
Channel.pageviewTracking = channelWatchPageviewTracking;
Channel.requiresJsForInteractivity = true;

Channel.getInitialProps = (ctx) =>
  channelPathGetInitialProps(ctx, {
    ...pageHeadQueryVariables(ctx),
    isBrowser: isBrowser(),
  });

Channel.query = graphql`
  query Channel_Query($login: String!, $isBrowser: Boolean!, $url: String!) {
    channel: user(login: $login) {
      ...StreamPlayer_channel
      ...StreamInfoBox_channel
      id
      login
      displayName
      description
      broadcastSettings {
        id
        title
        game {
          displayName
        }
      }
      stream {
        id
        createdAt
        height
        previewImageURL
        game {
          id
          name
        }
      }
      hosting {
        id
        login
        stream {
          id
          game {
            id
            name
          }
        }
      }
    }
    channelHome: user(login: $login) @skip(if: $isBrowser) {
      ...ChannelHomePage_channel @relay(mask: false)
    }
    ...PageHead_query
  }
`;
