import { datatype, internet, lorem } from 'faker';
import Router from 'next/router';
import { GQL_GARBAGE_OBJECT_ID } from 'tachyon-relay';
import { createShallowWrapperFactory } from 'tachyon-test-utils';
import type { PartialDeep } from 'type-fest';
import { RouteName } from '../../../routing';
import { NotFoundErrorLayout } from '../../layouts';
import { ChannelHomePage } from '../ChannelHomePage';
import { ChannelPlayerLayout } from './ChannelPlayerLayout';
import type { Channel_QueryResponse } from './__generated__/Channel_Query.graphql';
import { Channel, shouldRenderChannelHome } from '.';

jest.mock('../../../workers/chat.worker', () => jest.fn());

describe(Channel, () => {
  let mockRouterPush: jest.SpyInstance;

  const setup = createShallowWrapperFactory(Channel, () => {
    const login = internet.userName();

    return {
      ' $fragmentRefs': {
        PageHead_query: true,
      },
      channel: {
        ' $fragmentRefs': {
          StreamInfoBox_channel: true,
          StreamPlayer_channel: true,
        },
        broadcastSettings: {
          game: {
            displayName: internet.userAgent(),
          },
          id: datatype.uuid(),
          title: lorem.words(),
        },
        description: lorem.words(),
        displayName: internet.userName(),
        hosting: null,
        id: datatype.uuid(),
        login,
        stream: {
          createdAt: '2019-06-04T12:04:36Z',
          game: {
            id: datatype.uuid(),
            name: lorem.word(),
          },
          height: datatype.number({ max: 2000, min: 1 }),
          id: datatype.uuid(),
          previewImageURL:
            'https://static-cdn.jtvnw.net/broadcaster/thumb-{width}x{height}.jpg',
        },
      },
      channelHome: null,
      queryVariables: {
        isBrowser: false,
        login,
        url: internet.url(),
      },
    };
  });

  describe('Channel is offline', () => {
    afterEach(() => {
      mockRouterPush.mockReset();
    });

    it('does not render ChannelHomePage when channel is online', () => {
      const { wrapper } = setup({
        channel: { stream: { id: datatype.uuid() } },
      });

      expect(wrapper.find(ChannelHomePage)).not.toExist();
    });

    describe('when initial request is server-side rendered', () => {
      it('renders ChannelHomePage instead of Channel', () => {
        const { wrapper } = setup({
          channel: { stream: null },
          channelHome: {} as any,
        });

        expect(wrapper.find(ChannelHomePage)).toExist();
      });

      it('falls back on trying to render the Channel page when channelHome is missing', () => {
        const { wrapper } = setup({
          channel: { stream: null },
        });

        expect(wrapper.find(ChannelPlayerLayout)).toExist();
      });
    });

    describe('when client-side navigating', () => {
      it('redirects', () => {
        const { wrapper } = setup({
          channel: { stream: null },
          queryVariables: { isBrowser: true },
        });

        expect(
          wrapper.find({ params: { route: RouteName.ChannelHome } }),
        ).toExist();
      });
    });
  });

  beforeEach(() => {
    mockRouterPush = jest.spyOn(Router, 'push');
  });

  it('renders a Stream Player when the channel is found', () => {
    const { wrapper } = setup();
    expect(wrapper.find(ChannelPlayerLayout)).toExist();
  });

  it('renders a not found page when the channel is not found', () => {
    const { wrapper } = setup({ channel: null });
    expect(wrapper.find(NotFoundErrorLayout)).toExist();
  });

  describe('static methods', () => {
    const login = internet.userName();

    function testChannelProps(
      opts: PartialDeep<Channel_QueryResponse['channel']> = {},
    ): any {
      return {
        id: datatype.uuid(),
        login,
        ...opts,
      };
    }

    describe(shouldRenderChannelHome, () => {
      it('returns false for non-existent channel', () => {
        expect(shouldRenderChannelHome(null)).toEqual(false);
      });

      it('returns true when stream or hosting is missing', () => {
        expect(shouldRenderChannelHome(testChannelProps())).toEqual(true);
      });

      it('returns true when stream id is missing', () => {
        expect(
          shouldRenderChannelHome(testChannelProps({ stream: {} })),
        ).toEqual(true);
      });

      it('returns true when stream id indicates garbage object', () => {
        expect(
          shouldRenderChannelHome(
            testChannelProps({
              stream: { id: GQL_GARBAGE_OBJECT_ID },
            }),
          ),
        ).toEqual(true);
      });

      it('returns false when stream id is valid', () => {
        expect(
          shouldRenderChannelHome(testChannelProps({ stream: { id: 'id' } })),
        ).toEqual(false);
      });

      it('returns true when hosting stream is missing', () => {
        expect(
          shouldRenderChannelHome(
            testChannelProps({
              hosting: { id: 'id', login: 'therealderekt', stream: null },
            }),
          ),
        ).toEqual(true);
      });

      it('returns true when hosting stream id is missing', () => {
        expect(
          shouldRenderChannelHome(
            testChannelProps({
              hosting: { id: 'id', login: 'therealderekt', stream: {} },
            }),
          ),
        ).toEqual(true);
      });

      it('returns true when hosting stream id indicates garbage object', () => {
        expect(
          shouldRenderChannelHome(
            testChannelProps({
              hosting: {
                id: 'id',
                login: 'therealderekt',
                stream: { id: GQL_GARBAGE_OBJECT_ID },
              },
            }),
          ),
        ).toEqual(true);
      });

      it('returns false when hosting stream id is valid', () => {
        expect(
          shouldRenderChannelHome(
            testChannelProps({
              hosting: {
                id: 'id',
                login: 'therealderekt',
                stream: { id: 'id' },
              },
              stream: null,
            }),
          ),
        ).toEqual(false);
      });
    });
  });
});
