import { graphql } from 'react-relay/hooks';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import type { ChannelPathParameters } from 'tachyon-page-utils';
import {
  channelAboutPageviewTracking,
  channelPathGetInitialProps,
} from 'tachyon-page-utils';
import { Column, Grid, GridGutterSize, Layout, SVGAsset } from 'twitch-core-ui';
import { ChannelEmptyState } from '../../common';
import { pageHeadQueryVariables } from '../../growth';
import { ChannelLayout } from '../../layouts';
import type { PageWithQueryProps, TomorrowPage } from '../types';
import { DefaultPanel } from './DefaultPanel';
import type {
  ChannelAboutPage_QueryResponse,
  ChannelAboutPage_QueryVariables,
} from './__generated__/ChannelAboutPage_Query.graphql';

type ChannelAboutPageInitialProps = {
  queryVariables: ChannelAboutPage_QueryVariables;
};

type ChannelAboutPageProps = PageWithQueryProps<
  ChannelAboutPage_QueryResponse,
  ChannelAboutPageInitialProps
>;

type AboutPanel = NonNullable<
  NonNullable<ChannelAboutPageProps['channel']>['panels'][0]
>;

const ScLayout = styled(Layout)`
  display: flex;
  margin: 0 auto;
  max-width: 68rem;
`;

export const ChannelAboutPage: TomorrowPage<
  ChannelAboutPageInitialProps,
  ChannelAboutPageProps,
  ChannelPathParameters
> = ({ channel, loading, queryVariables }) => {
  const { formatMessage } = useIntl();
  const panels = (channel?.panels ?? []).filter(
    (panel): panel is AboutPanel => panel?.type === 'DEFAULT',
  );

  return (
    <ChannelLayout
      loading={loading}
      login={queryVariables.login}
      user={channel}
    >
      {channel &&
        (panels.length === 0 ? (
          <ChannelEmptyState
            asset={SVGAsset.Account}
            message={formatMessage(
              'Streamers use this space to fully express themselves. Pop into their chat and ask them a question!',
              'ChannelAboutPage',
            )}
            title={formatMessage('Tell me more!', 'ChannelAboutPage')}
          />
        ) : (
          <ScLayout>
            <Grid gutterSize={GridGutterSize.Small}>
              {panels.map((panel) => (
                <Column cols={{ default: 12, sm: 6 }} key={panel.id}>
                  <DefaultPanel panel={panel} />
                </Column>
              ))}
            </Grid>
          </ScLayout>
        ))}
    </ChannelLayout>
  );
};

ChannelAboutPage.displayName = 'ChannelAboutPage';
ChannelAboutPage.pageviewTracking = channelAboutPageviewTracking;
// TODO: verify non-js interactivity after dev
ChannelAboutPage.requiresJsForInteractivity = false;
ChannelAboutPage.handlesLoading = true;
ChannelAboutPage.getInitialProps = (ctx) =>
  channelPathGetInitialProps(ctx, pageHeadQueryVariables(ctx));

ChannelAboutPage.query = graphql`
  query ChannelAboutPage_Query($login: String!, $url: String!) {
    channel: user(login: $login) {
      ...ChannelLayout_user
      id
      login
      stream {
        id
      }
      panels {
        type
        ... on DefaultPanel {
          id
          ...DefaultPanel_panel
        }
      }
    }
    ...PageHead_query
  }
`;
