import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  AlignItems,
  Button,
  ButtonType,
  Color,
  Display,
  DropDownMenu,
  DropDownMenuItem,
  FontSize,
  Layout,
  useDialogState,
} from 'twitch-core-ui';
import { CHANNEL_CLIPS_FILTER_QUERY_PARAM_KEY } from '../../../../config';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { getClipsFilter, getFilterLabel } from '../utils';
import type { ClipsPeriod } from './__generated__/ClipList_RefetchQuery.graphql';

// istanbul ignore next: trivial
export const FilterMenu: FC<{ login: string; period: ClipsPeriod }> = ({
  login,
  period,
}) => {
  const intl = useIntl();
  const { formatMessage } = intl;
  const { anchorProps, dialogProps } = useDialogState();
  const filterLabel = formatMessage('Filter by', 'ChannelClipsPage');

  return (
    <Layout alignItems={AlignItems.Center} display={Display.Flex}>
      <Layout
        color={Color.Base}
        fontSize={FontSize.Size6}
        margin={{ right: 0.5 }}
      >
        {filterLabel}:
      </Layout>
      <Button dropdown variant={ButtonType.Secondary} {...anchorProps}>
        {getFilterLabel(intl, period)}
      </Button>
      <DropDownMenu {...dialogProps}>
        <FilterMenuItem
          isSelected={period === 'LAST_DAY'}
          login={login}
          period="LAST_DAY"
        />
        <FilterMenuItem
          isSelected={period === 'LAST_WEEK'}
          login={login}
          period="LAST_WEEK"
        />
        <FilterMenuItem
          isSelected={period === 'LAST_MONTH'}
          login={login}
          period="LAST_MONTH"
        />
        <FilterMenuItem
          isSelected={period === 'ALL_TIME'}
          login={login}
          period="ALL_TIME"
        />
      </DropDownMenu>
    </Layout>
  );
};

FilterMenu.displayName = 'ClipsListFilterMenu';

// istanbul ignore next: trivial
const FilterMenuItem: FC<{
  isSelected: boolean;
  login: string;
  period: ClipsPeriod;
}> = ({ isSelected, login, period }) => {
  const intl = useIntl();
  return (
    <DropDownMenuItem
      linkTo="/deferToRenderLink"
      renderLink={renderTachyonLink({
        query: {
          [CHANNEL_CLIPS_FILTER_QUERY_PARAM_KEY]: getClipsFilter(period),
        },
        route: RouteName.ChannelClips,
        routeParams: { login },
      })}
      selected={isSelected}
    >
      {getFilterLabel(intl, period)}
    </DropDownMenuItem>
  );
};

FilterMenuItem.displayName = 'ClipsListFilterMenuItem';
