import type { FC } from 'react';
import { useCallback } from 'react';
import type { RelayRefetchProp } from 'react-relay/legacy';
import { createRefetchContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import { useIntl } from 'tachyon-intl';
import { useRefetchList } from 'tachyon-relay';
import {
  AlignItems,
  Color,
  CoreText,
  Display,
  FontSize,
  FontWeight,
  JustifyContent,
  Layout,
  SVGAsset,
  TextTransform,
} from 'twitch-core-ui';
import { ChannelEmptyState, ClipCard, InfiniteList } from '../../../common';
import {
  CLIP_LIST_LOAD_MORE_LIMIT,
  CLIP_LIST_PAGE_SIZE,
  getEmptyStateTitle,
  getListTitle,
} from '../utils';
import { FilterMenu } from './FilterMenu';
import type { ClipsPeriod } from './__generated__/ClipList_RefetchQuery.graphql';
import type { ClipList_user } from './__generated__/ClipList_user.graphql';

type ClipListProps = {
  period: ClipsPeriod;
  relay: RelayRefetchProp;
  user: ClipList_user;
};

export const ScInfiniteList = styled(InfiniteList)`
  display: flex;
  flex-direction: column;
`;

export const ClipListBase: FC<ClipListProps> = ({ period, relay, user }) => {
  const intl = useIntl();
  const { formatMessage } = intl;

  const {
    endCursor: cursor,
    noMore,
    nodes: items,
  } = useRefetchList(user.clips?.edges);

  const itemRenderer = useCallback(
    (index: number) => {
      const clip = items[index];
      return <ClipCard clip={clip} key={clip.id} />;
    },
    [items],
  );

  const loadMore = useCallback(() => {
    if (noMore) {
      return;
    }

    relay.refetch({
      cursor,
      limit: CLIP_LIST_LOAD_MORE_LIMIT,
      login: user.login,
      period,
    });
  }, [cursor, noMore, period, relay, user.login]);

  return (
    <>
      <Layout
        alignItems={AlignItems.Center}
        display={Display.Flex}
        fontSize={FontSize.Size5}
        fontWeight={FontWeight.SemiBold}
        justifyContent={JustifyContent.Between}
        margin={{ bottom: 1 }}
      >
        <CoreText color={Color.Alt2} transform={TextTransform.Uppercase}>
          {getListTitle(intl, period)}
        </CoreText>
        <FilterMenu login={user.login} period={period} />
      </Layout>
      {items.length === 0 ? (
        <ChannelEmptyState
          asset={SVGAsset.Clip}
          message={formatMessage(
            "It's quiet... too quiet...",
            'ChannelClipsPage',
          )}
          title={getEmptyStateTitle(intl, period)}
        />
      ) : (
        <ScInfiniteList
          itemRenderer={itemRenderer}
          length={items.length}
          loadMore={loadMore}
          pageSize={CLIP_LIST_PAGE_SIZE}
        />
      )}
    </>
  );
};

ClipListBase.displayName = 'ClipList';

export const ClipList = createRefetchContainer(
  ClipListBase,
  {
    user: graphql`
      fragment ClipList_user on User {
        id
        login
        clips(first: $limit, after: $cursor, criteria: { period: $period }) {
          pageInfo {
            hasNextPage
          }
          edges {
            cursor
            node {
              id
              ...ClipCard_clip
            }
          }
        }
      }
    `,
  },
  graphql`
    query ClipList_RefetchQuery(
      $login: String!
      $limit: Int!
      $cursor: Cursor
      $period: ClipsPeriod
    ) {
      user(login: $login) {
        ...ClipList_user
      }
    }
  `,
);
