import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import type { ChannelPageTrackingParams } from 'tachyon-discovery';
import {
  ChannelContentType,
  ChannelSection,
  ChannelSubsection,
  useDiscoveryTracking,
} from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { getFormattedUserDisplayName, uniqueIDGenerator } from 'tachyon-utils';
import { Layout } from 'twitch-core-ui';
import { HomePageShelf } from '../HomePageShelf';
import { GameCard } from './GameCard';
import type { CategoryShelf_items } from './__generated__/CategoryShelf_items.graphql';

type CategoryShelfProps = {
  items: CategoryShelf_items;
  user: {
    displayName: string;
    id: string;
    login: string;
  };
};

const ScCardContainer = styled(Layout)`
  width: 19rem;
`;

export const CategoryShelfBase: FC<CategoryShelfProps> = ({ items, user }) => {
  const { formatMessage } = useIntl();
  const { onChannelPageItemClick, onChannelPageItemDisplay } =
    useDiscoveryTracking();

  const formattedDisplayName = getFormattedUserDisplayName(user);
  const title = formatMessage(
    "{formattedDisplayName}'s recently streamed categories",
    { formattedDisplayName },
    'CategoryShelf',
  );

  return (
    <HomePageShelf title={title}>
      {items.map((game, itemPosition) => {
        const trackingProps: ChannelPageTrackingParams = {
          category: game.name,
          channelID: user.id,
          contentType: ChannelContentType.Category,
          itemPosition,
          itemSubsection: ChannelSubsection.RecentCategories,
          itemTrackingID: uniqueIDGenerator(16),
          login: user.login,
          modelTrackingID: null,
          section: ChannelSection.Home,
        };
        return (
          <ScCardContainer flexShrink={0} key={game.id} margin={{ x: 0.5 }}>
            <GameCard
              game={game}
              onClick={() => {
                onChannelPageItemClick(trackingProps);
              }}
              onDisplay={() => {
                onChannelPageItemDisplay(trackingProps);
              }}
            />
          </ScCardContainer>
        );
      })}
    </HomePageShelf>
  );
};

CategoryShelfBase.displayName = 'CategoryShelf';

export const CategoryShelf = createFragmentContainer(CategoryShelfBase, {
  items: graphql`
    fragment CategoryShelf_items on Game @relay(plural: true) {
      id
      name
      ...GameCard_game
    }
  `,
});
