import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { formatLiveViewersCount } from 'tachyon-more-ui';
import { getFormattedUserDisplayName } from 'tachyon-utils';
import {
  Color,
  CoreLinkType,
  CoreText,
  Display,
  JustifyContent,
  Layout,
  MediaCard,
  MediaCardLink,
  MediaCardMeta,
  TextAlign,
} from 'twitch-core-ui';
import { RouteName, renderTachyonLink } from '../../../../routing';
import { CardLink, ChannelAvatar } from '../../../common';
import type { UserCard_user } from './__generated__/UserCard_user.graphql';

type UserCardProps = {
  onClick: () => void;
  onDisplay: () => void;
  user: UserCard_user;
};

// istanbul ignore next: trivial
export const UserCardBase: FC<UserCardProps> = ({
  onClick,
  onDisplay,
  user,
}) => {
  const { formatMessage, formatNumberShort } = useIntl();

  let statusText: string | null;
  if (user.stream) {
    statusText =
      user.stream.viewersCount !== null
        ? formatLiveViewersCount(
            formatMessage,
            user.stream.viewersCount,
            formatNumberShort(user.stream.viewersCount, {
              maximumSignificantDigits: 2,
            }),
          )
        : null;
  } else {
    statusText = formatMessage('Offline', 'StreamerShelf');
  }

  return (
    <CardLink
      linkTo="/deferToRenderLink"
      onClick={onClick}
      onImpression={onDisplay}
      renderLink={renderTachyonLink({
        route: RouteName.Channel,
        routeParams: { login: user.login },
      })}
      variant={CoreLinkType.Inherit}
    >
      <Layout fullWidth>
        <MediaCard
          image={
            <Layout
              display={Display.Flex}
              justifyContent={JustifyContent.Center}
            >
              <ChannelAvatar
                isLive={!!user.stream}
                login={user.login}
                primaryColorHex={user.primaryColorHex}
                profileImageURL={user.profileImageURL}
              />
            </Layout>
          }
          meta={
            <MediaCardMeta
              links={
                <Layout textAlign={TextAlign.Center}>
                  <CoreText color={Color.Alt2} lines={1}>
                    {getFormattedUserDisplayName(user)}
                  </CoreText>
                  {statusText !== null && (
                    <MediaCardLink>{statusText}</MediaCardLink>
                  )}
                </Layout>
              }
            />
          }
        />
      </Layout>
    </CardLink>
  );
};

UserCardBase.displayName = 'UserCard';

export const UserCard = createFragmentContainer(UserCardBase, {
  user: graphql`
    fragment UserCard_user on User {
      id
      displayName
      login
      stream {
        viewersCount
      }
      primaryColorHex
      profileImageURL(width: 150)
    }
  `,
});
