import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import type { ChannelPageTrackingParams } from 'tachyon-discovery';
import {
  ChannelContentType,
  ChannelSection,
  ChannelSubsection,
  useDiscoveryTracking,
} from 'tachyon-discovery';
import { useIntl } from 'tachyon-intl';
import { getFormattedUserDisplayName, uniqueIDGenerator } from 'tachyon-utils';
import { Layout } from 'twitch-core-ui';
import { HomePageShelf } from '../HomePageShelf';
import { UserCard } from './UserCard';
import type { StreamerShelf_items } from './__generated__/StreamerShelf_items.graphql';

type StreamerShelfProps = {
  items: StreamerShelf_items;
  user: {
    displayName: string;
    id: string;
    login: string;
  };
};

const ScCardContainer = styled(Layout)`
  width: 10.5rem;
`;

export const StreamerShelfBase: FC<StreamerShelfProps> = ({ items, user }) => {
  const { formatMessage } = useIntl();
  const { onChannelPageItemClick, onChannelPageItemDisplay } =
    useDiscoveryTracking();

  const formattedDisplayName = getFormattedUserDisplayName(user);
  const title = formatMessage(
    '{formattedDisplayName} suggests these streamers',
    { formattedDisplayName },
    'StreamerShelf',
  );

  return (
    <HomePageShelf title={title}>
      {items.map((streamer, itemPosition) => {
        const trackingProps: ChannelPageTrackingParams = {
          category: null,
          channelID: user.id,
          contentType: ChannelContentType.LiveStream,
          itemPosition,
          itemSubsection: ChannelSubsection.FeaturedStreamer,
          itemTrackingID: uniqueIDGenerator(16),
          login: user.login,
          modelTrackingID: null,
          section: ChannelSection.Home,
        };

        return (
          <ScCardContainer flexShrink={0} key={streamer.id} margin={{ x: 0.5 }}>
            <UserCard
              onClick={() => {
                onChannelPageItemClick({
                  ...trackingProps,
                  component: 'user_thumbnail',
                });
              }}
              onDisplay={() => {
                onChannelPageItemDisplay(trackingProps);
              }}
              user={streamer}
            />
          </ScCardContainer>
        );
      })}
    </HomePageShelf>
  );
};

StreamerShelfBase.displayName = 'StreamerShelf';

export const StreamerShelf = createFragmentContainer(StreamerShelfBase, {
  items: graphql`
    fragment StreamerShelf_items on User @relay(plural: true) {
      id
      ...UserCard_user
    }
  `,
});
