import type { FC } from 'react';
import { useIntl } from 'tachyon-intl';
import {
  formatTotalViewCount,
  getChannelThumbnailSrcAndSrcSet,
} from 'tachyon-more-ui';
import { formatVideoLength } from 'tachyon-utils';
import {
  AspectRatio,
  CardImage,
  CardImageSize,
  MediaCard,
  MediaCardImage,
  MediaCardLink,
  MediaCardMeta,
  MediaCardStat,
  MediaCardTitle,
} from 'twitch-core-ui';

// Convenience type for representing data associated with a video. Clips and VODs
// have a slightly different shape coming from GraphQL, but this forces them
// into a standard metadata object.
type VideoCardMeta = {
  createdAt: string | null;
  game: {
    displayName: string;
  } | null;
  lengthSeconds: number | null;
  previewThumbnailURL: string;
  title: string | null;
  viewCount: number | null;
};

type VideoShelfItemProps = {
  item: VideoCardMeta;
};

// istanbul ignore next: trivial
export const VideoShelfItem: FC<VideoShelfItemProps> = ({ item }) => {
  const { formatMessage, formatNumberShort, formatPastRelativeDate } =
    useIntl();
  const gameName = item.game?.displayName;
  const viewCount = item.viewCount
    ? formatTotalViewCount(
        formatMessage,
        item.viewCount,
        formatNumberShort(item.viewCount, {
          maximumSignificantDigits: 2,
        }),
      )
    : null;

  const createdAt = item.createdAt
    ? formatPastRelativeDate(new Date(item.createdAt))
    : null;

  let subtitle: string | null = null;
  if (gameName) {
    subtitle = gameName;
    if (viewCount) {
      subtitle = `${subtitle} · ${viewCount}`;
    }
    if (createdAt) {
      subtitle = `${subtitle} · ${createdAt}`;
    }
  }

  return (
    <MediaCard
      image={
        <MediaCardImage
          image={
            <CardImage
              aspect={AspectRatio.Aspect16x9}
              size={CardImageSize.Size2}
              {...getChannelThumbnailSrcAndSrcSet(item.previewThumbnailURL)}
              alt=""
            />
          }
          ratio={AspectRatio.Aspect16x9}
          topLeft={
            item.lengthSeconds ? (
              <MediaCardStat>
                {formatVideoLength(item.lengthSeconds)}
              </MediaCardStat>
            ) : null
          }
        />
      }
      meta={
        <MediaCardMeta
          links={
            <>{subtitle ? <MediaCardLink>{subtitle}</MediaCardLink> : null}</>
          }
          title={<MediaCardTitle lines={2}>{item.title ?? ''}</MediaCardTitle>}
        />
      }
    />
  );
};

VideoShelfItem.displayName = 'VideoShelfItem';
