import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import styled from 'styled-components';
import type { ChannelPageTrackingParams } from 'tachyon-discovery';
import {
  ChannelContentType,
  ChannelSection,
  ChannelSubsection,
  useDiscoveryTracking,
} from 'tachyon-discovery';
import { uniqueIDGenerator } from 'tachyon-utils';
import { Layout } from 'twitch-core-ui';
import { HomePageShelf } from '../HomePageShelf';
import { VideoShelfClip } from './VideoShelfClip';
import { VideoShelfVideo } from './VideoShelfVideo';
import type { VideoShelf_items } from './__generated__/VideoShelf_items.graphql';

type VideoShelfProps = {
  items: VideoShelf_items;
  title: string;
  user: {
    id: string;
    login: string;
  };
};

const ScCardContainer = styled(Layout)`
  width: 29rem;
`;

const getCategory = (item: VideoShelf_items[0]): string | null => {
  switch (item.__typename) {
    case 'Clip':
    case 'Video':
      return item.game?.name ?? null;
    default:
      return null;
  }
};

export const VideoShelfBase: FC<VideoShelfProps> = ({ items, title, user }) => {
  const { onChannelPageItemClick, onChannelPageItemDisplay } =
    useDiscoveryTracking();

  return (
    <HomePageShelf title={title}>
      {items.map((item, index) => {
        const baseTrackingProps: ChannelPageTrackingParams = {
          category: getCategory(item),
          channelID: user.id,
          contentType: ChannelContentType.Clip,
          itemPosition: index,
          itemSubsection: ChannelSubsection.RecentBroadcasts,
          itemTrackingID: uniqueIDGenerator(16),
          login: user.login,
          modelTrackingID: null,
          section: ChannelSection.Home,
        };

        return (
          <ScCardContainer flexShrink={0} key={index} margin={{ x: 0.5 }}>
            {item.__typename === 'Clip' ? (
              <VideoShelfClip
                clip={item}
                onClick={() => {
                  onChannelPageItemClick({
                    ...baseTrackingProps,
                    contentType: ChannelContentType.Clip,
                  });
                }}
                onDisplay={() => {
                  onChannelPageItemDisplay({
                    ...baseTrackingProps,
                    contentType: ChannelContentType.Clip,
                  });
                }}
              />
            ) : item.__typename === 'Video' ? (
              <VideoShelfVideo
                onClick={() => {
                  onChannelPageItemClick({
                    ...baseTrackingProps,
                    contentType: ChannelContentType.Video,
                  });
                }}
                onDisplay={() => {
                  onChannelPageItemDisplay({
                    ...baseTrackingProps,
                    contentType: ChannelContentType.Video,
                  });
                }}
                video={item}
              />
            ) : null}
          </ScCardContainer>
        );
      })}
    </HomePageShelf>
  );
};

VideoShelfBase.displayName = 'VideoShelf';

export const VideoShelf = createFragmentContainer(VideoShelfBase, {
  items: graphql`
    fragment VideoShelf_items on VideoShelfItem @relay(plural: true) {
      __typename
      ... on Clip {
        id
        game {
          name
        }
        ...VideoShelfClip_clip
      }
      ... on Video {
        id
        game {
          name
        }
        ...VideoShelfVideo_video
      }
    }
  `,
});
