import { graphql } from 'react-relay/hooks';
import { useIntl } from 'tachyon-intl';
import type { ChannelPathParameters } from 'tachyon-page-utils';
import {
  channelHomePageviewTracking,
  channelPathGetInitialProps,
} from 'tachyon-page-utils';
import { reduceToNonNullNodes } from 'tachyon-utils';
import { SVGAsset } from 'twitch-core-ui';
import { ChannelEmptyState } from '../../common';
import { pageHeadQueryVariables } from '../../growth';
import { ChannelLayout } from '../../layouts';
import type { PageWithQueryProps, TomorrowPage } from '../types';
import { CategoryShelf } from './CategoryShelf';
import { FeaturedSection } from './FeaturedSection';
import { StreamerShelf } from './StreamerShelf';
import { VideoShelf } from './VideoShelf';
import type {
  ChannelHomePage_QueryResponse,
  ChannelHomePage_QueryVariables,
} from './__generated__/ChannelHomePage_Query.graphql';

type ChannelHomePageInitialProps = {
  queryVariables: ChannelHomePage_QueryVariables;
};

type ChannelHomePageProps = PageWithQueryProps<
  ChannelHomePage_QueryResponse,
  ChannelHomePageInitialProps
>;

export const ChannelHomePage: TomorrowPage<
  ChannelHomePageInitialProps,
  ChannelHomePageProps,
  ChannelPathParameters
> = ({ channel, loading, queryVariables }) => {
  const { formatMessage } = useIntl();
  const videoShelf = channel?.videoShelves?.edges?.[0]?.node;
  const categoryShelf = channel?.channel?.home?.shelves?.categoryShelf;
  const streamerShelf = channel?.channel?.home?.shelves?.streamerShelf;

  const stream = channel?.stream ?? null;
  const hosting = channel?.hosting?.stream ?? null;
  const archives = reduceToNonNullNodes(channel?.archives?.edges);
  const videos = (videoShelf?.items ?? []).filter(Boolean);
  const games = reduceToNonNullNodes(categoryShelf?.edges);
  const streamers = reduceToNonNullNodes(streamerShelf?.edges);

  const hasContent =
    !!stream ||
    !!hosting ||
    archives.length > 0 ||
    videos.length > 0 ||
    games.length > 0 ||
    streamers.length > 0;

  return (
    <ChannelLayout
      loading={loading}
      login={queryVariables.login}
      user={channel}
    >
      {channel &&
        (!hasContent ? (
          <ChannelEmptyState
            asset={SVGAsset.NavChannels}
            message={formatMessage(
              "It's quiet... too quiet",
              'ChannelHomePage',
            )}
            title={formatMessage(
              'This channel has no content',
              'ChannelHomePage',
            )}
          />
        ) : (
          <>
            <FeaturedSection
              archives={archives}
              hosting={hosting}
              stream={stream}
            />
            {videoShelf && videos.length > 0 && (
              <VideoShelf
                items={videos}
                title={videoShelf.title}
                user={channel}
              />
            )}

            {categoryShelf && games.length > 0 && (
              <CategoryShelf items={games} user={channel} />
            )}
            {streamerShelf && streamers.length > 0 && (
              <StreamerShelf items={streamers} user={channel} />
            )}
          </>
        ))}
    </ChannelLayout>
  );
};

ChannelHomePage.displayName = 'ChannelHomePage';
ChannelHomePage.pageviewTracking = channelHomePageviewTracking;
// TODO: verify non-js interactivity after dev
ChannelHomePage.requiresJsForInteractivity = false;
ChannelHomePage.handlesLoading = true;
ChannelHomePage.getInitialProps = (ctx) =>
  channelPathGetInitialProps(ctx, pageHeadQueryVariables(ctx));

// eslint-disable-next-line no-unused-expressions
graphql`
  fragment ChannelHomePage_channel on User {
    ...ChannelLayout_user
    id
    login
    displayName
    stream {
      id
      ...FeaturedSection_stream
    }
    hosting {
      stream {
        ...FeaturedSection_hosting
      }
    }
    archives: videos(first: 1, type: ARCHIVE, sort: TIME) {
      edges {
        node {
          ...FeaturedSection_archives
        }
      }
    }
    videoShelves(first: 1) {
      edges {
        node {
          title
          items {
            ...VideoShelf_items
          }
        }
      }
    }
    channel {
      home {
        shelves {
          categoryShelf {
            edges {
              node {
                ...CategoryShelf_items
              }
            }
          }
          streamerShelf {
            edges {
              node {
                ...StreamerShelf_items
              }
            }
          }
        }
      }
    }
  }
`;

ChannelHomePage.query = graphql`
  query ChannelHomePage_Query($login: String!, $url: String!) {
    channel: user(login: $login) {
      ...ChannelHomePage_channel @relay(mask: false)
    }
    ...PageHead_query
  }
`;
