import type { FC } from 'react';
import { createFragmentContainer, graphql } from 'react-relay/legacy';
import { useIntl } from 'tachyon-intl';
import { groupBy } from 'tachyon-utils';
import { Layout } from 'twitch-core-ui';
import { ScheduleSegmentCard } from './ScheduleSegmentCard';
import { ScheduleTitle } from './ScheduleTitle';
import type { FutureSegments_interruption } from './__generated__/FutureSegments_interruption.graphql';
import type { FutureSegments_segments } from './__generated__/FutureSegments_segments.graphql';
import { getTimeRange } from './utils';

type FutureSegmentsProps = {
  interruption: FutureSegments_interruption | null;
  segments: FutureSegments_segments;
};

// istanbul ignore next: trivial
export const FutureSegmentsBase: FC<FutureSegmentsProps> = ({
  interruption,
  segments,
}) => {
  const { formatDate } = useIntl();

  const futureSegments = groupBy(segments, (segment) =>
    formatDate(new Date(segment.startAt), {
      day: 'numeric',
      month: 'long',
      weekday: 'long',
    }),
  );

  return (
    <>
      {Object.entries(futureSegments).map(([title, items]) => {
        return (
          <Layout key={title}>
            <Layout padding={{ bottom: 2 }}>
              <ScheduleTitle>{title}</ScheduleTitle>
            </Layout>
            {items.map((segment) => (
              <ScheduleSegmentCard
                interruption={getTimeRange(interruption)}
                key={segment.id}
                segment={segment}
              />
            ))}
          </Layout>
        );
      })}
    </>
  );
};

FutureSegmentsBase.displayName = 'FutureSegments';

export const FutureSegments = createFragmentContainer(FutureSegmentsBase, {
  interruption: graphql`
    fragment FutureSegments_interruption on ScheduleInterruption {
      startAt
      endAt
    }
  `,
  segments: graphql`
    fragment FutureSegments_segments on ScheduleSegment @relay(plural: true) {
      id
      startAt
      ...ScheduleSegmentCard_segment
    }
  `,
});
